#ifndef DISCRETE_FUNCTION_DPK_VECTOR_HPP
#define DISCRETE_FUNCTION_DPK_VECTOR_HPP

#include <language/utils/ASTNodeDataTypeTraits.hpp>

#include <mesh/ItemArray.hpp>
#include <mesh/ItemArrayUtils.hpp>
#include <mesh/MeshData.hpp>
#include <mesh/MeshDataManager.hpp>
#include <mesh/MeshVariant.hpp>
#include <scheme/DiscreteFunctionDescriptorP0.hpp>
#include <scheme/PolynomialCenteredCanonicalBasisView.hpp>

template <size_t Dimension,
          typename DataType,
          typename BasisView = PolynomialCenteredCanonicalBasisView<Dimension, DataType>>
class DiscreteFunctionDPkVector
{
 public:
  using data_type = DataType;

  friend class DiscreteFunctionDPkVector<Dimension, std::add_const_t<DataType>>;
  friend class DiscreteFunctionDPkVector<Dimension, std::remove_const_t<DataType>>;

  class ComponentCoefficientView
  {
   private:
    DataType* const m_data;
    const size_t m_size;

   public:
    size_t
    size() const
    {
      return m_size;
    }

    DataType&
    operator[](size_t i) const
    {
      Assert(i < m_size);
      return m_data[i];
    }

    ComponentCoefficientView(DataType* data, size_t size) : m_data{data}, m_size{size} {}

    ComponentCoefficientView(const ComponentCoefficientView&) = delete;
    ComponentCoefficientView(ComponentCoefficientView&&)      = delete;

    ~ComponentCoefficientView() = default;
  };

 private:
  std::shared_ptr<const MeshVariant> m_mesh_v;
  size_t m_degree;
  size_t m_number_of_components;
  size_t m_nb_coefficients_per_component;
  CellArray<DataType> m_by_cell_coefficients;
  CellValue<const TinyVector<Dimension>> m_xj;

 public:
  PUGS_INLINE
  ASTNodeDataType
  dataType() const
  {
    return ast_node_data_type_from<std::remove_const_t<DataType>>;
  }

  PUGS_INLINE size_t
  degree() const
  {
    return m_degree;
  }

  PUGS_INLINE size_t
  numberOfComponents() const
  {
    return m_number_of_components;
  }

  PUGS_INLINE
  size_t
  numberOfCoefficientsPerComponent() const
  {
    return m_nb_coefficients_per_component;
  }

  PUGS_INLINE
  const CellArray<DataType>&
  cellArrays() const
  {
    return m_by_cell_coefficients;
  }

  PUGS_INLINE std::shared_ptr<const MeshVariant>
  meshVariant() const
  {
    return m_mesh_v;
  }

  PUGS_FORCEINLINE
  operator DiscreteFunctionDPkVector<Dimension, const DataType>() const
  {
    return DiscreteFunctionDPkVector<Dimension, const DataType>(m_mesh_v, m_degree, m_number_of_components,
                                                                m_by_cell_coefficients);
  }

  PUGS_INLINE
  void
  fill(const DataType& data) const noexcept
  {
    static_assert(not std::is_const_v<DataType>, "Cannot modify ItemValue of const");
    m_by_cell_coefficients.fill(data);
  }

  friend PUGS_INLINE DiscreteFunctionDPkVector<Dimension, std::remove_const_t<DataType>>
  copy(const DiscreteFunctionDPkVector& source)
  {
    return DiscreteFunctionDPkVector<Dimension, std::remove_const_t<DataType>>{source.m_mesh_v, source.m_degree,
                                                                               source.m_number_of_components,
                                                                               copy(source.cellArrays())};
  }

  friend PUGS_INLINE void
  copy_to(const DiscreteFunctionDPkVector<Dimension, DataType>& source,
          DiscreteFunctionDPkVector<Dimension, std::remove_const_t<DataType>>& destination)
  {
    Assert(source.m_mesh_v->id() == destination.m_mesh_v->id(), "copy_to target must use the same mesh");
    Assert(source.m_degree == destination.m_degree, "copy_to target must have the same degree");
    Assert(source.m_number_of_components == destination.m_number_of_components,
           "copy_to target must have the same number of components");
    copy_to(source.m_by_cell_coefficients, destination.m_by_cell_coefficients);
  }

  PUGS_INLINE
  auto
  coefficients(const CellId cell_id) const
  {
    return m_by_cell_coefficients[cell_id];
  }

  PUGS_INLINE auto
  componentCoefficients(const CellId cell_id, size_t i_component) const noexcept(NO_ASSERT)
  {
    Assert(m_mesh_v.use_count() > 0, "DiscreteFunctionDPkVector is not built");
    Assert(i_component < m_number_of_components, "incorrect component number");

    return ComponentCoefficientView{&m_by_cell_coefficients[cell_id][i_component * m_nb_coefficients_per_component],
                                    m_nb_coefficients_per_component};
  }

  PUGS_FORCEINLINE BasisView
  operator()(const CellId cell_id, size_t i_component) const noexcept(NO_ASSERT)
  {
    Assert(m_mesh_v.use_count() > 0, "DiscreteFunctionDPkVector is not built");
    Assert(i_component < m_number_of_components, "incorrect component number");
    ComponentCoefficientView
      component_coefficient_view{&m_by_cell_coefficients[cell_id][i_component * m_nb_coefficients_per_component],
                                 m_nb_coefficients_per_component};
    return BasisView{m_degree, component_coefficient_view, m_xj[cell_id]};
  }

  PUGS_INLINE DiscreteFunctionDPkVector
  operator=(const DiscreteFunctionDPkVector& f)
  {
    m_mesh_v                        = f.m_mesh_v;
    m_degree                        = f.m_degree;
    m_number_of_components          = f.m_number_of_components;
    m_nb_coefficients_per_component = f.m_nb_coefficients_per_component;
    m_by_cell_coefficients          = f.m_by_cell_coefficients;
    return *this;
  }

  DiscreteFunctionDPkVector(const std::shared_ptr<const MeshVariant>& mesh_v,
                            size_t degree,
                            size_t number_of_components)
    : m_mesh_v{mesh_v},
      m_degree{degree},
      m_number_of_components{number_of_components},
      m_nb_coefficients_per_component(BasisView::dimensionFromDegree(degree)),
      m_by_cell_coefficients{mesh_v->connectivity(), m_number_of_components * BasisView::dimensionFromDegree(degree)},
      m_xj{MeshDataManager::instance().getMeshData(*m_mesh_v->get<Mesh<Dimension>>()).xj()}
  {}

  DiscreteFunctionDPkVector(const std::shared_ptr<const MeshVariant>& mesh_v,
                            size_t degree,
                            size_t number_of_components,
                            CellArray<DataType> by_cell_coefficients)
    : m_mesh_v{mesh_v},
      m_degree{degree},
      m_number_of_components{number_of_components},
      m_nb_coefficients_per_component(BasisView::dimensionFromDegree(degree)),
      m_by_cell_coefficients{by_cell_coefficients},
      m_xj{MeshDataManager::instance().getMeshData(*m_mesh_v->get<Mesh<Dimension>>()).xj()}
  {
    Assert(mesh_v->connectivity().id() == by_cell_coefficients.connectivity_ptr()->id(),
           "cell_array is built on different connectivity");
  }

  template <MeshConcept MeshType>
  DiscreteFunctionDPkVector(const std::shared_ptr<const MeshType>& p_mesh, size_t degree, size_t number_of_components)
    : DiscreteFunctionDPkVector{p_mesh->meshVariant(), degree, number_of_components}
  {}

  template <MeshConcept MeshType>
  DiscreteFunctionDPkVector(const std::shared_ptr<const MeshType>& p_mesh,
                            size_t degree,
                            size_t number_of_components,
                            CellArray<DataType> by_cell_coefficients)
    : DiscreteFunctionDPkVector{p_mesh->meshVariant(), degree, number_of_components, by_cell_coefficients}
  {}

  DiscreteFunctionDPkVector() noexcept = default;

  DiscreteFunctionDPkVector(const DiscreteFunctionDPkVector&) noexcept = default;
  DiscreteFunctionDPkVector(DiscreteFunctionDPkVector&&) noexcept      = default;

  ~DiscreteFunctionDPkVector() = default;
};

#endif   // DISCRETE_FUNCTION_DPK_VECTOR_HPP
