#ifndef BOUNDARY_CONDITION_HANDLER_HPP
#define BOUNDARY_CONDITION_HANDLER_HPP

#include <vector>
#include <memory>

class BoundaryCondition
{
public:
  enum Type {
    velocity,
    normal_velocity,
    pressure,
    symmetry
  };

  const Type m_type;
protected:

  BoundaryCondition(const Type& type)
    : m_type(type)
  {
    ;
  }

public:
  const Type& type() const
  {
    return m_type;
  }

  virtual ~BoundaryCondition() = default;
};

class PressureBoundaryCondition
  : public BoundaryCondition
{
private:
  const double m_value;

  const size_t m_number_of_faces;
  Kokkos::View<unsigned int*> m_face_list;

public:
  double value() const
  {
    return m_value;
  }

  const size_t& numberOfFaces() const
  {
    return m_number_of_faces;
  }

  const Kokkos::View<const unsigned int*> faceList() const
  {
    return m_face_list;
  }

  PressureBoundaryCondition(const double& value,
			    const std::vector<unsigned int>& faces)
    : BoundaryCondition(BoundaryCondition::pressure),
      m_value(value),
      m_number_of_faces(faces.size()),
      m_face_list("faces_list", m_number_of_faces)
  {
    Kokkos::parallel_for(m_number_of_faces, KOKKOS_LAMBDA(const int& f){
	m_face_list[f]=faces[f];
      });
  }

  ~PressureBoundaryCondition() = default;
};

template <size_t dimension>
class SymmetryBoundaryCondition
  : public BoundaryCondition
{
public:
  typedef TinyVector<dimension, double> Rd;

private:
  Kokkos::View<unsigned int*> m_node_list;
  const Rd m_outgoing_normal;
public:
  const Rd& outgoingNormal() const
  {
    return m_outgoing_normal;
  }
  size_t numberOfNodes() const
  {
    return m_node_list.size();
  }

  const Kokkos::View<const unsigned int*> nodeList() const
  {
    return m_node_list;
  }

  SymmetryBoundaryCondition(const std::vector<unsigned int>& nodes,
			    const Rd& outgoing_normal)
    : BoundaryCondition(BoundaryCondition::symmetry),
      m_node_list("node_list", nodes.size()),
      m_outgoing_normal(outgoing_normal)
  {
    Kokkos::parallel_for(m_node_list.size(), KOKKOS_LAMBDA(const int& f){
	m_node_list[f]=nodes[f];
      });
  }

  ~SymmetryBoundaryCondition() = default;
};

class BoundaryConditionHandler
{
private:
  std::shared_ptr<BoundaryCondition> m_boundary_condition;

public:
  const BoundaryCondition& boundaryCondition() const
  {
    return *m_boundary_condition;
  }
  
  KOKKOS_INLINE_FUNCTION
  BoundaryConditionHandler& operator=(BoundaryConditionHandler&) = default;

  KOKKOS_INLINE_FUNCTION
  BoundaryConditionHandler(const BoundaryConditionHandler&) = default;

  KOKKOS_INLINE_FUNCTION
  BoundaryConditionHandler(BoundaryConditionHandler&&) = default;

  KOKKOS_INLINE_FUNCTION
  BoundaryConditionHandler(std::shared_ptr<BoundaryCondition> boundary_condition)
    : m_boundary_condition(boundary_condition)
  {
  }

  ~BoundaryConditionHandler() = default;
};

#endif // BOUNDARY_CONDITION_HANDLER_HPP
