#ifndef SYMBOL_TABLE_HPP
#define SYMBOL_TABLE_HPP

#include <ASTNodeDataType.hpp>
#include <ASTNodeDataVariant.hpp>

#include <pegtl/position.hpp>

#include <iostream>

class SymbolTable
{
 public:
  class Attributes
  {
   private:
    bool m_is_initialized{false};
    ASTNodeDataType m_data_type{ASTNodeDataType::undefined_t};
    ASTNodeDataVariant m_value;

    TAO_PEGTL_NAMESPACE::position m_position;

   public:
    auto&
    value()
    {
      return m_value;
    }

    const auto&
    value() const
    {
      return m_value;
    }

    const bool&
    isInitialized() const
    {
      return m_is_initialized;
    }

    void
    setIsInitialized()
    {
      m_is_initialized = true;
    }

    const ASTNodeDataType&
    dataType() const
    {
      return m_data_type;
    }

    auto
    position() const
    {
      return m_position;
    }

    void
    setDataType(const ASTNodeDataType& data_type)
    {
      m_data_type = data_type;
    }

    friend std::ostream&
    operator<<(std::ostream& os, const Attributes& attributes)
    {
      std::visit(
        [&](const auto& value) {
          using T = std::decay_t<decltype(value)>;
          if constexpr (std::is_same_v<T, std::monostate>) {
            os << "--";
          } else {
            os << value;
          }
        },
        attributes.m_value);

      return os;
    }

    Attributes(const TAO_PEGTL_NAMESPACE::position& position) : m_position(position) {}

    Attributes(const Attributes&) = default;
  };

 private:
  std::vector<std::pair<std::string, Attributes>> m_symbol_list;
  std::shared_ptr<SymbolTable> m_parent_table;

 public:
  friend std::ostream&
  operator<<(std::ostream& os, const SymbolTable& symbol_table)
  {
    os << "-- Symbol table state -- parent : " << symbol_table.m_parent_table.get() << "\n";
    for (auto i_symbol : symbol_table.m_symbol_list) {
      os << ' ' << i_symbol.first << ": " << std::boolalpha << i_symbol.second << '\n';
    }
    os << "------------------------\n";
    return os;
  }

  auto
  find(const std::string& symbol, const TAO_PEGTL_NAMESPACE::position& use_position)
  {
    auto i_symbol = m_symbol_list.end();

    for (auto i_stored_symbol = m_symbol_list.begin(); i_stored_symbol != m_symbol_list.end(); ++i_stored_symbol) {
      if (i_stored_symbol->first == symbol) {
        i_symbol = i_stored_symbol;
        break;
      }
    }

    if (i_symbol != m_symbol_list.end() and (use_position.byte >= i_symbol->second.position().byte)) {
      return std::make_pair(i_symbol, true);
    } else {
      if (m_parent_table) {
        return m_parent_table->find(symbol, use_position);
      } else {
        return std::make_pair(i_symbol, false);
      }
    }
  }

  auto
  add(const std::string& symbol, const TAO_PEGTL_NAMESPACE::position& symbol_position)
  {
    for (auto i_stored_symbol = m_symbol_list.begin(); i_stored_symbol != m_symbol_list.end(); ++i_stored_symbol) {
      if (i_stored_symbol->first == symbol) {
        return std::make_pair(i_stored_symbol, false);
      }
    }
    return std::make_pair(m_symbol_list.emplace(m_symbol_list.end(),
                                                std::make_pair(symbol, Attributes(symbol_position))),
                          true);
  }

  SymbolTable(const std::shared_ptr<SymbolTable>& parent_table = nullptr) : m_parent_table(parent_table)
  {
    ;
  }
};

#endif   // SYMBOL_TABLE_HPP
