#include <ASTNodeUnaryOperatorExpressionBuilder.hpp>
#include <PEGGrammar.hpp>
#include <SymbolTable.hpp>

namespace language
{
template <typename Op>
struct UnaryOp;

template <>
struct UnaryOp<language::unary_minus>
{
  template <typename A>
  PUGS_INLINE A
  eval(const A& a)
  {
    return -a;
  }
};

template <>
struct UnaryOp<language::unary_not>
{
  template <typename A>
  PUGS_INLINE bool
  eval(const A& a)
  {
    return not a;
  }
};

template <typename UnaryOpT, typename ValueT, typename DataT>
class UnaryExpressionProcessor final : public INodeProcessor
{
  ASTNode& m_node;

 public:
  PUGS_INLINE ValueT
  eval(const ASTNodeDataVariant& a)
  {
    return UnaryOp<UnaryOpT>().eval(std::get<DataT>(a));
  }

 public:
  UnaryExpressionProcessor(ASTNode& node) : m_node{node} {}

  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    m_node.children[0]->execute(exec_policy);
    m_node.m_value = eval(m_node.children[0]->m_value);
  }
};

ASTNodeUnaryOperatorExpressionBuilder::ASTNodeUnaryOperatorExpressionBuilder(ASTNode& n)
{
  auto set_unary_operator_processor = [](ASTNode& n, const auto& operator_v) {
    auto set_unary_operator_processor_for_data = [&](const auto& value, const ASTNodeDataType& data_type) {
      using OperatorT = std::decay_t<decltype(operator_v)>;
      using ValueT    = std::decay_t<decltype(value)>;
      switch (data_type) {
      case ASTNodeDataType::bool_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, bool>>(n);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, uint64_t>>(n);
        break;
      }
      case ASTNodeDataType::int_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, int64_t>>(n);
        break;
      }
      case ASTNodeDataType::double_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, double>>(n);
        break;
      }
      default: {
        throw parse_error("undefined operand type for unary operator", std::vector{n.children[0]->begin()});
      }
      }
    };

    auto set_unary_operator_processor_for_value = [&](const ASTNodeDataType& value_type) {
      const ASTNodeDataType data_type = n.children[0]->m_data_type;
      switch (value_type) {
      case ASTNodeDataType::bool_t: {
        set_unary_operator_processor_for_data(bool{}, data_type);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        set_unary_operator_processor_for_data(uint64_t{}, data_type);
        break;
      }
      case ASTNodeDataType::int_t: {
        set_unary_operator_processor_for_data(int64_t{}, data_type);
        break;
      }
      case ASTNodeDataType::double_t: {
        set_unary_operator_processor_for_data(double{}, data_type);
        break;
      }
      default: {
        throw parse_error("undefined value type for unary operator", std::vector{n.begin()});
      }
      }
    };

    set_unary_operator_processor_for_value(n.m_data_type);
  };

  if (n.is<language::unary_minus>()) {
    set_unary_operator_processor(n, language::unary_minus{});
  } else if (n.is<language::unary_not>()) {
    set_unary_operator_processor(n, language::unary_not{});
  } else {
    throw parse_error("unexpected error: undefined unary operator", std::vector{n.begin()});
  }
}
}   // namespace language
