#include <ASTNodeAffectationExpressionBuilder.hpp>
#include <PEGGrammar.hpp>
#include <SymbolTable.hpp>

namespace language
{
template <typename Op>
struct AffOp;

template <>
struct AffOp<language::eq_op>
{
  template <typename A, typename B>
  PUGS_INLINE void
  eval(A& a, const B& b)
  {
    a = b;
  }
};

template <>
struct AffOp<language::multiplyeq_op>
{
  template <typename A, typename B>
  PUGS_INLINE void
  eval(A& a, const B& b)
  {
    a *= b;
  }
};

template <>
struct AffOp<language::divideeq_op>
{
  template <typename A, typename B>
  PUGS_INLINE void
  eval(A& a, const B& b)
  {
    a /= b;
  }
};

template <>
struct AffOp<language::pluseq_op>
{
  template <typename A, typename B>
  PUGS_INLINE void
  eval(A& a, const B& b)
  {
    a += b;
  }
};

template <>
struct AffOp<language::minuseq_op>
{
  template <typename A, typename B>
  PUGS_INLINE void
  eval(A& a, const B& b)
  {
    a -= b;
  }
};

template <typename OperatorT, typename ValueT, typename DataT>
class AffectationProcessor final : public INodeProcessor
{
 private:
  ASTNode& m_node;
  ASTNodeDataVariant* p_value{nullptr};

  static inline const bool _is_defined{[] {
    if constexpr (std::is_same_v<std::decay_t<ValueT>, bool>) {
      if constexpr (not std::is_same_v<OperatorT, language::eq_op>) {
        return false;
      }
    }
    return true;
  }()};

 public:
  AffectationProcessor(ASTNode& node) : m_node{node}
  {
    if constexpr (_is_defined) {
      const std::string& symbol = m_node.children[0]->string();
      auto [i_symbol, found]    = m_node.m_symbol_table->find(symbol, m_node.children[0]->begin());
      Assert(found);
      p_value = &i_symbol->second.value();
    } else {
      throw parse_error("invalid operands to binary expression", std::vector{m_node.begin()});
    }
  }

  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    if constexpr (_is_defined) {
      m_node.children[1]->execute(exec_policy);

      if constexpr (std::is_same_v<OperatorT, language::eq_op>) {
        if constexpr (std::is_same_v<ValueT, DataT>) {
          *p_value = m_node.children[1]->m_value;
        } else {
          *p_value = static_cast<ValueT>(std::get<DataT>(m_node.children[1]->m_value));
        }
      } else {
        AffOp<OperatorT>().eval(std::get<ValueT>(*p_value), std::get<DataT>(m_node.children[1]->m_value));
      }
    }
  }
};

template <typename OperatorT, typename DataT>
class AffectationToStringProcessor final : public INodeProcessor
{
 private:
  ASTNode& m_node;
  ASTNodeDataVariant* p_value{nullptr};

 public:
  AffectationToStringProcessor(ASTNode& node) : m_node{node}
  {
    const std::string& symbol = m_node.children[0]->string();
    auto [i_symbol, found]    = m_node.m_symbol_table->find(symbol, m_node.children[0]->begin());
    Assert(found);
    p_value = &i_symbol->second.value();
  }

  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    m_node.children[1]->execute(exec_policy);

    if constexpr (std::is_same_v<OperatorT, language::eq_op>) {
      if constexpr (std::is_same_v<std::string, DataT>) {
        *p_value = m_node.children[1]->m_value;
      } else {
        *p_value = std::to_string(std::get<DataT>(m_node.children[1]->m_value));
      }
    } else if constexpr (std::is_same_v<OperatorT, language::pluseq_op>) {
      if constexpr (std::is_same_v<std::string, DataT>) {
        std::get<std::string>(*p_value) += std::get<std::string>(m_node.children[1]->m_value);
      } else {
        std::get<std::string>(*p_value) += std::to_string(std::get<DataT>(m_node.children[1]->m_value));
      }
    }
  }
};   // namespace language

ASTNodeAffectationExpressionBuilder::ASTNodeAffectationExpressionBuilder(ASTNode& n)
{
  auto set_affectation_processor = [](ASTNode& n, const auto& operator_v) {
    auto set_affectation_processor_for_data = [&](const auto& value, const ASTNodeDataType& data_type) {
      using OperatorT = std::decay_t<decltype(operator_v)>;
      using ValueT    = std::decay_t<decltype(value)>;
      switch (data_type) {
      case ASTNodeDataType::bool_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, bool>>(n);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, uint64_t>>(n);
        break;
      }
      case ASTNodeDataType::int_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, int64_t>>(n);
        break;
      }
      case ASTNodeDataType::double_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, double>>(n);
        break;
      }
      default: {
        throw parse_error("undefined operand type for affectation", std::vector{n.children[0]->begin()});
      }
      }
    };

    auto set_affectation_processor_for_string_data = [&](const ASTNodeDataType& data_type) {
      using OperatorT = std::decay_t<decltype(operator_v)>;

      if constexpr (std::is_same_v<OperatorT, language::eq_op> or std::is_same_v<OperatorT, language::pluseq_op>) {
        switch (data_type) {
        case ASTNodeDataType::bool_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, bool>>(n);
          break;
        }
        case ASTNodeDataType::unsigned_int_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, uint64_t>>(n);
          break;
        }
        case ASTNodeDataType::int_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, int64_t>>(n);
          break;
        }
        case ASTNodeDataType::double_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, double>>(n);
          break;
        }
        case ASTNodeDataType::string_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, std::string>>(n);
          break;
        }
        default: {
          throw parse_error("undefined operand type for affectation", std::vector{n.children[0]->begin()});
        }
        }
      } else {
        throw parse_error("undefined operator type", std::vector{n.children[0]->begin()});
      }
    };

    auto set_affectation_processor_for_value = [&](const ASTNodeDataType& value_type) {
      const ASTNodeDataType data_type = n.children[1]->m_data_type;
      switch (value_type) {
      case ASTNodeDataType::bool_t: {
        set_affectation_processor_for_data(bool{}, data_type);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        set_affectation_processor_for_data(uint64_t{}, data_type);
        break;
      }
      case ASTNodeDataType::int_t: {
        set_affectation_processor_for_data(int64_t{}, data_type);
        break;
      }
      case ASTNodeDataType::double_t: {
        set_affectation_processor_for_data(double{}, data_type);
        break;
      }
      case ASTNodeDataType::string_t: {
        set_affectation_processor_for_string_data(data_type);
        break;
      }
      default: {
        throw parse_error("undefined value type for affectation", std::vector{n.begin()});
      }
      }
    };

    set_affectation_processor_for_value(n.m_data_type);
  };

  if (n.is<language::eq_op>()) {
    set_affectation_processor(n, language::eq_op{});
  } else if (n.is<language::multiplyeq_op>()) {
    set_affectation_processor(n, language::multiplyeq_op{});
  } else if (n.is<language::divideeq_op>()) {
    set_affectation_processor(n, language::divideeq_op{});
  } else if (n.is<language::pluseq_op>()) {
    set_affectation_processor(n, language::pluseq_op{});
  } else if (n.is<language::minuseq_op>()) {
    set_affectation_processor(n, language::minuseq_op{});
  } else {
    throw parse_error("unexpected error: undefined affectation operator", std::vector{n.begin()});
  }
}
}   // namespace language
