#include <language/modules/MeshModule.hpp>

#include <algebra/TinyVector.hpp>
#include <language/node_processor/ExecutionPolicy.hpp>
#include <language/utils/BuiltinFunctionEmbedder.hpp>
#include <language/utils/FunctionTable.hpp>
#include <language/utils/SymbolTable.hpp>
#include <language/utils/TypeDescriptor.hpp>
#include <mesh/CartesianMeshBuilder.hpp>
#include <mesh/Connectivity.hpp>
#include <mesh/DualMeshManager.hpp>
#include <mesh/GmshReader.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/MeshRelaxer.hpp>
#include <mesh/MeshTransformer.hpp>
#include <utils/Exceptions.hpp>

#include <Kokkos_Core.hpp>

MeshModule::MeshModule()
{
  this->_addTypeDescriptor(ast_node_data_type_from<std::shared_ptr<const IMesh>>);

  this->_addBuiltinFunction("readGmsh",
                            std::make_shared<BuiltinFunctionEmbedder<std::shared_ptr<const IMesh>(const std::string&)>>(

                              [](const std::string& file_name) -> std::shared_ptr<const IMesh> {
                                GmshReader gmsh_reader(file_name);
                                return gmsh_reader.mesh();
                              }

                              ));

  this->_addBuiltinFunction("transform",
                            std::make_shared<BuiltinFunctionEmbedder<
                              std::shared_ptr<const IMesh>(std::shared_ptr<const IMesh>, const FunctionSymbolId&)>>(

                              [](std::shared_ptr<const IMesh> p_mesh,
                                 const FunctionSymbolId& function_id) -> std::shared_ptr<const IMesh> {
                                return MeshTransformer{}.transform(function_id, p_mesh);
                              }

                              ));

  this->_addBuiltinFunction("relax",
                            std::make_shared<BuiltinFunctionEmbedder<
                              std::shared_ptr<const IMesh>(const std::shared_ptr<const IMesh>&,
                                                           const std::shared_ptr<const IMesh>&, const double&)>>(

                              [](const std::shared_ptr<const IMesh>& source_mesh,
                                 const std::shared_ptr<const IMesh>& destination_mesh,
                                 const double& theta) -> std::shared_ptr<const IMesh> {
                                return MeshRelaxer{}.relax(source_mesh, destination_mesh, theta);
                              }

                              ));

  this->_addBuiltinFunction("cartesian1dMesh",
                            std::make_shared<BuiltinFunctionEmbedder<std::shared_ptr<
                              const IMesh>(const TinyVector<1>, const TinyVector<1>, const std::vector<uint64_t>&)>>(

                              [](const TinyVector<1> a, const TinyVector<1> b,
                                 const std::vector<uint64_t>& box_sizes) -> std::shared_ptr<const IMesh> {
                                constexpr uint64_t dimension = 1;

                                if (box_sizes.size() != dimension) {
                                  throw NormalError("expecting " + std::to_string(dimension) +
                                                    " dimensions, provided " + std::to_string(box_sizes.size()));
                                }

                                const TinyVector<dimension, uint64_t> sizes = [&]() {
                                  TinyVector<dimension, uint64_t> s;
                                  for (size_t i = 0; i < dimension; ++i) {
                                    s[i] = box_sizes[i];
                                  }
                                  return s;
                                }();

                                CartesianMeshBuilder builder{a, b, sizes};
                                return builder.mesh();
                              }

                              ));

  this->_addBuiltinFunction("cartesian2dMesh",
                            std::make_shared<BuiltinFunctionEmbedder<std::shared_ptr<
                              const IMesh>(const TinyVector<2>, const TinyVector<2>, const std::vector<uint64_t>&)>>(

                              [](const TinyVector<2> a, const TinyVector<2> b,
                                 const std::vector<uint64_t>& box_sizes) -> std::shared_ptr<const IMesh> {
                                constexpr uint64_t dimension = 2;

                                if (box_sizes.size() != dimension) {
                                  throw NormalError("expecting " + std::to_string(dimension) +
                                                    " dimensions, provided " + std::to_string(box_sizes.size()));
                                }

                                const TinyVector<dimension, uint64_t> sizes = [&]() {
                                  TinyVector<dimension, uint64_t> s;
                                  for (size_t i = 0; i < dimension; ++i) {
                                    s[i] = box_sizes[i];
                                  }
                                  return s;
                                }();

                                CartesianMeshBuilder builder{a, b, sizes};
                                return builder.mesh();
                              }

                              ));

  this->_addBuiltinFunction("cartesian3dMesh",
                            std::make_shared<BuiltinFunctionEmbedder<std::shared_ptr<
                              const IMesh>(const TinyVector<3>&, const TinyVector<3>&, const std::vector<uint64_t>&)>>(

                              [](const TinyVector<3>& a, const TinyVector<3>& b,
                                 const std::vector<uint64_t>& box_sizes) -> std::shared_ptr<const IMesh> {
                                constexpr uint64_t dimension = 3;

                                if (box_sizes.size() != dimension) {
                                  throw NormalError("expecting " + std::to_string(dimension) +
                                                    " dimensions, provided " + std::to_string(box_sizes.size()));
                                }

                                const TinyVector<dimension, uint64_t> sizes = [&]() {
                                  TinyVector<dimension, uint64_t> s;
                                  for (size_t i = 0; i < dimension; ++i) {
                                    s[i] = box_sizes[i];
                                  }
                                  return s;
                                }();

                                CartesianMeshBuilder builder{a, b, sizes};
                                return builder.mesh();
                              }

                              ));

  this->_addBuiltinFunction("diamondDual",
                            std::make_shared<BuiltinFunctionEmbedder<std::shared_ptr<const IMesh>(
                              const std::shared_ptr<const IMesh>&)>>(

                              [](const std::shared_ptr<const IMesh>& i_mesh) -> std::shared_ptr<const IMesh> {
                                switch (i_mesh->dimension()) {
                                case 1: {
                                  using MeshType = Mesh<Connectivity<1>>;

                                  std::shared_ptr p_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);
                                  return DualMeshManager::instance().getDiamondDualMesh(*p_mesh);
                                }
                                case 2: {
                                  using MeshType = Mesh<Connectivity<2>>;

                                  std::shared_ptr p_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);
                                  return DualMeshManager::instance().getDiamondDualMesh(*p_mesh);
                                }
                                case 3: {
                                  using MeshType = Mesh<Connectivity<3>>;

                                  std::shared_ptr p_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);
                                  return DualMeshManager::instance().getDiamondDualMesh(*p_mesh);
                                }
                                default: {
                                  throw UnexpectedError("invalid dimension");
                                }
                                }
                              }

                              ));

  this->_addBuiltinFunction("medianDual",
                            std::make_shared<BuiltinFunctionEmbedder<std::shared_ptr<const IMesh>(
                              const std::shared_ptr<const IMesh>&)>>(

                              [](const std::shared_ptr<const IMesh>& i_mesh) -> std::shared_ptr<const IMesh> {
                                switch (i_mesh->dimension()) {
                                case 1: {
                                  using MeshType = Mesh<Connectivity<1>>;

                                  std::shared_ptr p_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);
                                  return DualMeshManager::instance().getMedianDualMesh(*p_mesh);
                                }
                                case 2: {
                                  using MeshType = Mesh<Connectivity<2>>;

                                  std::shared_ptr p_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);
                                  return DualMeshManager::instance().getMedianDualMesh(*p_mesh);
                                }
                                case 3: {
                                  using MeshType = Mesh<Connectivity<3>>;

                                  std::shared_ptr p_mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);
                                  return DualMeshManager::instance().getMedianDualMesh(*p_mesh);
                                }
                                default: {
                                  throw UnexpectedError("invalid dimension");
                                }
                                }
                              }

                              ));
}

void
MeshModule::registerOperators() const
{}
