#include <utils/SignalManager.hpp>

#include <utils/BacktraceManager.hpp>
#include <utils/ConsoleManager.hpp>
#include <utils/Messenger.hpp>
#include <utils/PugsAssert.hpp>

#include <utils/pugs_config.hpp>

#include <rang.hpp>

#include <csignal>
#include <iostream>
#include <unistd.h>

bool SignalManager::s_pause_on_error = false;

bool
SignalManager::pauseOnError()
{
  return s_pause_on_error;
}

void
SignalManager::setPauseForDebug(bool pause_on_error)
{
  s_pause_on_error = pause_on_error;
}

std::string
SignalManager::signalName(int signal)
{
  switch (signal) {
  case SIGILL:
    return "SIGILL";
  case SIGFPE:
    return "SIGFPE";
  case SIGABRT:
    return "SIGABRT";
  case SIGINT:
    return "SIGINT";
  case SIGSEGV:
    return "SIGSEGV";
  case SIGTERM:
    return "SIGTERM";
  }
  return "SIGNAL undefined!";
}

void
SignalManager::pauseForDebug(int signal)
{
  if (std::string(PUGS_BUILD_TYPE) != "Release") {
    if (s_pause_on_error) {
      std::cerr << "\n======================================\n"
                << rang::style::reset << rang::fg::reset << rang::style::bold << "to attach gdb to this process run\n"
                << "\tgdb -pid " << rang::fg::red << getpid() << rang::fg::reset << '\n'
                << "else press Control-C to exit\n"
                << rang::style::reset << "======================================\n"
                << std::flush;
      pause();
    }
  }
  std::exit(signal);
}

void
SignalManager::handler(int signal)
{
  std::signal(SIGFPE, SIG_DFL);
  std::signal(SIGSEGV, SIG_IGN);
  std::signal(SIGTERM, SIG_DFL);
  std::signal(SIGINT, SIG_DFL);
  std::signal(SIGABRT, SIG_DFL);

  // Each failing process must write
  std::cerr.setstate(std::ios::goodbit);

  BacktraceManager bm;
  std::cerr << bm << '\n';

  std::exception_ptr eptr = std::current_exception();
  try {
    if (eptr) {
      std::rethrow_exception(eptr);
    }
  }
  catch (const NotImplementedError& not_implemented_error) {
    std::cerr << not_implemented_error.what() << '\n';
  }
  catch (const UnexpectedError& unexpected_error) {
    std::cerr << unexpected_error.what() << '\n';
  }
  catch (const AssertError& assert_error) {
    std::cerr << assert_error << '\n';
  }
  catch (...) {
    std::cerr << "Unknown exception!\n";
  }

  std::cerr << "\n *** " << rang::style::reset << rang::fg::reset << rang::style::bold << "Signal " << rang::fgB::red
            << signalName(signal) << rang::fg::reset << " caught" << rang::style::reset << " ***\n";

  SignalManager::pauseForDebug(signal);
}

void
SignalManager::init(bool enable)
{
  if (enable) {
    std::signal(SIGFPE, SignalManager::handler);
    std::signal(SIGSEGV, SignalManager::handler);
    std::signal(SIGTERM, SignalManager::handler);
    std::signal(SIGINT, SignalManager::handler);
    std::signal(SIGABRT, SignalManager::handler);
    std::signal(SIGPIPE, SignalManager::handler);
  }
}
