#include <PastisUtils.hpp>
#include <PastisOStream.hpp>

#include <Kokkos_Core.hpp>

#include <RevisionInfo.hpp>
#include <BuildInfo.hpp>

#include <rang.hpp>

#include <FPEManager.hpp>
#include <SignalManager.hpp>
#include <ConsoleManager.hpp>

#include <CLI/CLI.hpp>

#include <pastis_config.hpp>

#ifdef PASTIS_HAS_MPI
#include <mpi.h>
#endif // PASTIS_HAS_MPI

std::string initialize(int& argc, char* argv[])
{
#ifdef PASTIS_HAS_MPI
  MPI_Init(&argc, &argv);

  {
    const int mpi_rank
        =[](){
           int mpi_rank;
           MPI_Comm_rank(MPI_COMM_WORLD, &mpi_rank);
           return mpi_rank;
         }();
    if (mpi_rank != 0) {
      pout.setOutput(null_stream);
    }
  }
#endif // PASTIS_HAS_MPI

  long unsigned number = 10;
  std::string filename;

  pout() << "Pastis version: "
            << rang::style::bold << RevisionInfo::version() << rang::style::reset << '\n';

  pout() << "-------------------- "
            << rang::fg::green
            << "git info"
            << rang::fg::reset
            <<" -------------------------"
            << '\n';
  pout() << "tag:  " << rang::style::bold << RevisionInfo::gitTag() << rang::style::reset << '\n';
  pout() << "HEAD: " << rang::style::bold << RevisionInfo::gitHead() << rang::style::reset << '\n';
  pout() << "hash: " << rang::style::bold << RevisionInfo::gitHash() << rang::style::reset << "  (";

  if (RevisionInfo::gitIsClean()) {
    pout() << rang::fgB::green << "clean" << rang::fg::reset;
  } else {
    pout() << rang::fgB::red << "dirty" << rang::fg::reset;
  }
  pout() << ")\n";
  pout() << "-------------------- "
            << rang::fg::green
            << "build info"
            << rang::fg::reset
            <<" -----------------------"
            << '\n';
  pout() << "type:     " << rang::style::bold << BuildInfo::type() << rang::style::reset << '\n';
  pout() << "compiler: " << rang::style::bold << BuildInfo::compiler() << rang::style::reset << '\n';
  pout() << "kokkos:   " << rang::style::bold << BuildInfo::kokkosDevices() << rang::style::reset << '\n';
  pout() << "mpi:      " << rang::style::bold << BuildInfo::mpiLibrary() << rang::style::reset << '\n';
  pout() << "-------------------------------------------------------\n";
  {
    CLI::App app{"Pastis help"};

    app.add_option("-n,--number", number, "Number of cells");//->required();

    app.add_option("filename,-f,--filename", filename, "gmsh file");//->required();

    int threads=-1;
    app.add_option("--threads", threads, "Number of Kokkos threads")->check(CLI::Range(1,std::numeric_limits<decltype(threads)>::max()));

    std::string colorize="auto";
    app.add_set("--colorize", colorize, {"auto", "yes", "no"}, "Colorize console output", true);

    bool disable_fpe = false;
    app.add_flag("--no-fpe", disable_fpe, "Do not trap floating point exceptions");
    bool disable_signals = false;
    app.add_flag("--no-signal", disable_signals, "Do not catches signals");

    std::string pause_on_error="auto";
    app.add_set("--pause-on-error", pause_on_error, {"auto", "yes", "no"}, "Pause for debugging on unexpected error", true);

    std::atexit([](){pout() << rang::style::reset;});
    try {
      app.parse(argc, argv);
    } catch (const CLI::ParseError &e) {
      std::exit(app.exit(e));
    }

    ConsoleManager::init(colorize);
    FPEManager::init(not disable_fpe);
    SignalManager::setPauseForDebug(pause_on_error);
    SignalManager::init(not disable_signals);
  }

  Kokkos::initialize(argc,argv);
  pout() << "-------------------- "
            << rang::fg::green
            << "exec info"
            << rang::fg::reset
            <<" ------------------------"
            << '\n';

  pout() << rang::style::bold;
  Kokkos::DefaultExecutionSpace::print_configuration(pout());
  pout() << rang::style::reset;
  pout() << "-------------------------------------------------------\n";

  return filename;
}

void finalize()
{
#ifdef PASTIS_HAS_MPI
  MPI_Barrier(MPI_COMM_WORLD);
  const int mpi_rank
      =[](){
         int mpi_rank;
         MPI_Comm_rank(MPI_COMM_WORLD, &mpi_rank);
         return mpi_rank;
       }();

  const int mpi_size
      =[](){
         int mpi_size;
         MPI_Comm_size(MPI_COMM_WORLD, &mpi_size);
         return mpi_size;
       }();

  pout() << rang::fgB::green << "Terminating process " << rang::fg::reset
         << rang::fgB::yellow << mpi_rank << rang::fg::reset << " of "
         << rang::style::bold << mpi_size << rang::style::reset << '\n';;
#endif // PASTIS_HAS_MPI

  Kokkos::finalize();

#ifdef PASTIS_HAS_MPI
  MPI_Finalize();
#endif // PASTIS_HAS_MPI
}
