#ifndef AST_NODE_HPP
#define AST_NODE_HPP

#include <language/node_processor/ExecutionPolicy.hpp>
#include <language/node_processor/INodeProcessor.hpp>
#include <language/utils/ASTNodeDataType.hpp>
#include <language/utils/DataVariant.hpp>
#include <utils/PugsAssert.hpp>
#include <utils/PugsMacros.hpp>

#include <pegtl/contrib/parse_tree.hpp>

using namespace TAO_PEGTL_NAMESPACE;

class SymbolTable;

class ASTNode : public parse_tree::basic_node<ASTNode>
{
 private:
  PUGS_INLINE
  decltype(m_end)
  _getEnd() const
  {
    if (not this->has_content()) {
      if (this->children.size() > 0) {
        return this->children[children.size() - 1]->_getEnd();
      }
    }
    return m_end;
  }

  PUGS_INLINE
  decltype(m_begin)
  _getBegin() const
  {
    if (not this->has_content()) {
      if (this->children.size() > 0) {
        return this->children[0]->_getBegin();
      }
    }
    return m_begin;
  }

 public:
  std::shared_ptr<SymbolTable> m_symbol_table;
  std::unique_ptr<INodeProcessor> m_node_processor;

  ASTNodeDataType m_data_type;

  [[nodiscard]] PUGS_INLINE std::string
  string() const
  {
    if (this->has_content()) {
      return this->parse_tree::basic_node<ASTNode>::string();
    } else {
      auto end   = this->_getEnd();
      auto begin = this->_getBegin();
      if (end.data != nullptr) {
        return std::string{begin.data, end.data};
      }
      return {"<optimized out>"};
    }
  }

  [[nodiscard]] PUGS_INLINE std::string_view
  string_view() const
  {
    if (this->has_content()) {
      return this->parse_tree::basic_node<ASTNode>::string_view();
    } else {
      auto end   = this->_getEnd();
      auto begin = this->_getBegin();
      if (end.data != nullptr) {
        return std::string_view(begin.data, end.data - begin.data);
      }
      return {"<optimized out>"};
    }
  }

  PUGS_INLINE
  std::string
  name() const noexcept
  {
    return demangle(std::string{this->type});
  }

  PUGS_INLINE
  DataVariant
  execute(ExecutionPolicy& exec_policy)
  {
    Assert(m_node_processor, "undefined node processor");
    if (exec_policy.exec()) {
      return m_node_processor->execute(exec_policy);
    } else {
      return {};
    }
  }
};

#endif   // AST_NODE_HPP
