#ifndef RESUME_UTILS_HPP
#define RESUME_UTILS_HPP

#include <utils/HighFivePugsUtils.hpp>

#include <language/utils/SymbolTable.hpp>
#include <mesh/CellType.hpp>
#include <mesh/ItemArray.hpp>
#include <mesh/ItemValue.hpp>
#include <utils/Messenger.hpp>

template <typename DataType>
PUGS_INLINE Array<DataType>
readArray(const HighFive::Group& group, const std::string& name)
{
  auto get_address = [](auto& x) { return (x.size() > 0) ? &(x[0]) : nullptr; };

  using data_type = std::remove_const_t<DataType>;

  auto dataset = group.getDataSet(name);

  std::vector<size_t> size_per_rank = dataset.getAttribute("size_per_rank").read<std::vector<size_t>>();

  if (size_per_rank.size() != parallel::size()) {
    throw NormalError("cannot change number of processes");
  }

  std::vector<size_t> offset{0, 0ul};
  for (size_t i = 0; i < parallel::rank(); ++i) {
    offset[0] += size_per_rank[i];
  }
  std::vector<size_t> count{size_per_rank[parallel::rank()]};

  Array<DataType> array(size_per_rank[parallel::rank()]);
  if constexpr (std::is_same_v<CellType, data_type>) {
    using base_type = std::underlying_type_t<CellType>;
    dataset.select(offset, count).read_raw(reinterpret_cast<base_type*>(get_address(array)));
  } else if constexpr ((std::is_same_v<CellId, data_type>) or (std::is_same_v<FaceId, data_type>) or
                       (std::is_same_v<EdgeId, data_type>) or (std::is_same_v<NodeId, data_type>)) {
    using base_type = typename data_type::base_type;
    dataset.select(offset, count).read_raw(reinterpret_cast<base_type*>(get_address(array)));
  } else {
    dataset.select(offset, count).read_raw(get_address(array));
  }

  return array;
}

template <typename DataType>
PUGS_INLINE Table<DataType>
readTable(const HighFive::Group& group, const std::string& name)
{
  auto get_address = [](auto& t) { return (t.numberOfRows() * t.numberOfColumns() > 0) ? &(t(0, 0)) : nullptr; };

  using data_type = std::remove_const_t<DataType>;

  auto dataset = group.getDataSet(name);

  const size_t number_of_columns = dataset.getAttribute("number_of_columns").read<size_t>();
  const std::vector<size_t> number_of_rows_per_rank =
    dataset.getAttribute("number_of_rows_per_rank").read<std::vector<size_t>>();

  std::vector<size_t> offset{0, 0ul};
  for (size_t i = 0; i < parallel::rank(); ++i) {
    offset[0] += number_of_rows_per_rank[i] * number_of_columns;
  }
  std::vector<size_t> count{number_of_rows_per_rank[parallel::rank()]};

  Table<DataType> table(number_of_rows_per_rank[parallel::rank()], number_of_columns);
  if constexpr (std::is_same_v<CellType, data_type>) {
    using base_type = std::underlying_type_t<CellType>;
    dataset.select(offset, count).read_raw(reinterpret_cast<base_type*>(get_address(table)));
  } else if constexpr ((std::is_same_v<CellId, data_type>) or (std::is_same_v<FaceId, data_type>) or
                       (std::is_same_v<EdgeId, data_type>) or (std::is_same_v<NodeId, data_type>)) {
    using base_type = typename data_type::base_type;
    dataset.select(offset, count).read_raw<base_type>(reinterpret_cast<base_type*>(get_address(table)));
  } else {
    dataset.select(offset, count).read_raw<data_type>(get_address(table));
  }

  return table;
}

EmbeddedData readDiscreteFunctionVariant(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readIBoundaryConditionDescriptor(const std::string& symbol_name,
                                              const HighFive::Group& symbol_table_group);
EmbeddedData readIBoundaryDescriptor(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readIDiscreteFunctionDescriptor(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readIInterfaceDescriptor(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readINamedDiscreteData(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readIQuadratureDescriptor(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readItemArrayVariant(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readItemType(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readItemValueVariant(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readIZoneDescriptor(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readMesh(const std::string& symbol_name, const HighFive::Group& symbol_table_group);
EmbeddedData readOStream(const std::string& symbol_name, const HighFive::Group& symbol_table_group);

#endif   // RESUME_UTILS_HPP
