#ifndef IMPLICIT_ACOUSTIC_SOLVER_HPP
#define IMPLICIT_ACOUSTIC_SOLVER_HPP

#include <mesh/MeshTraits.hpp>

#include <memory>
#include <tuple>
#include <vector>

class DiscreteFunctionVariant;
class MeshVariant;
class ItemValueVariant;
class SubItemValuePerItemVariant;
class IBoundaryConditionDescriptor;
class IZoneDescriptor;

double acoustic_dt(const std::shared_ptr<const DiscreteFunctionVariant>& c,
                   const std::vector<std::shared_ptr<const IZoneDescriptor>>& explicit_zone_list);

std::shared_ptr<const DiscreteFunctionVariant> local_acoustic_dt(
  const std::shared_ptr<const DiscreteFunctionVariant>& c);

class ImplicitAcousticSolverHandler
{
 public:
  enum class SolverType
  {
    Glace1State,
    Glace2States,
    Eucclhyd
  };

 private:
  struct IImplicitAcousticSolver
  {
    virtual std::tuple<std::shared_ptr<const MeshVariant>,
                       std::shared_ptr<const DiscreteFunctionVariant>,
                       std::shared_ptr<const DiscreteFunctionVariant>,
                       std::shared_ptr<const DiscreteFunctionVariant>>
    apply(const double& dt,
          const std::shared_ptr<const DiscreteFunctionVariant>& rho,
          const std::shared_ptr<const DiscreteFunctionVariant>& u,
          const std::shared_ptr<const DiscreteFunctionVariant>& E,
          const std::shared_ptr<const DiscreteFunctionVariant>& c,
          const std::shared_ptr<const DiscreteFunctionVariant>& p,
          const std::shared_ptr<const DiscreteFunctionVariant>& pi,
          const std::shared_ptr<const DiscreteFunctionVariant>& gamma,
          const std::shared_ptr<const DiscreteFunctionVariant>& Cv,
          const std::shared_ptr<const DiscreteFunctionVariant>& entropy) = 0;

    IImplicitAcousticSolver()                                     = default;
    IImplicitAcousticSolver(IImplicitAcousticSolver&&)            = default;
    IImplicitAcousticSolver& operator=(IImplicitAcousticSolver&&) = default;

    virtual ~IImplicitAcousticSolver() = default;
  };

  template <MeshConcept MeshType>
  class ImplicitAcousticSolver;

  std::unique_ptr<IImplicitAcousticSolver> m_implicit_acoustic_solver;

 public:
  IImplicitAcousticSolver&
  solver()
  {
    return *m_implicit_acoustic_solver;
  }

  ImplicitAcousticSolverHandler(
    SolverType solver_type,
    const std::shared_ptr<const DiscreteFunctionVariant>& rho,
    const std::shared_ptr<const DiscreteFunctionVariant>& c,
    const std::shared_ptr<const DiscreteFunctionVariant>& u,
    const std::shared_ptr<const DiscreteFunctionVariant>& p,
    const std::shared_ptr<const DiscreteFunctionVariant>& pi,
    const std::shared_ptr<const DiscreteFunctionVariant>& gamma,
    const std::shared_ptr<const DiscreteFunctionVariant>& Cv,
    const std::shared_ptr<const DiscreteFunctionVariant>& entropy,
    const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list,
    const std::vector<std::shared_ptr<const IZoneDescriptor>>& explicit_zone_list,
    const double& dt);

  ImplicitAcousticSolverHandler(
    const SolverType solver_type,
    const std::shared_ptr<const DiscreteFunctionVariant>& rho,
    const std::shared_ptr<const DiscreteFunctionVariant>& c,
    const std::shared_ptr<const DiscreteFunctionVariant>& u,
    const std::shared_ptr<const DiscreteFunctionVariant>& p,
    const std::shared_ptr<const DiscreteFunctionVariant>& pi,
    const std::shared_ptr<const DiscreteFunctionVariant>& gamma,
    const std::shared_ptr<const DiscreteFunctionVariant>& Cv,
    const std::shared_ptr<const DiscreteFunctionVariant>& entropy,
    const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list,
    const std::shared_ptr<const DiscreteFunctionVariant>& chi_explicit,
    const double& dt);
};

#endif   // IMPLICIT_ACOUSTIC_SOLVER_HPP
