#include <algebra/LinearSolver.hpp>
#include <utils/pugs_config.hpp>

#include <algebra/BiCGStab.hpp>
#include <algebra/CG.hpp>
#include <algebra/PETScUtils.hpp>

#ifdef PUGS_HAS_PETSC
#include <petsc.h>
#endif   // PUGS_HAS_PETSC

struct LinearSolver::Internals
{
  static bool
  hasLibrary(const LSLibrary library)
  {
    switch (library) {
    case LSLibrary::builtin: {
      return true;
    }
    case LSLibrary::petsc: {
#ifdef PUGS_HAS_PETSC
      return true;
#else
      return false;
#endif
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("Linear system library (" + ::name(library) + ") was not set!");
    }
      // LCOV_EXCL_STOP
    }
  }

  static void
  checkHasLibrary(const LSLibrary library)
  {
    if (not hasLibrary(library)) {
      // LCOV_EXCL_START
      throw NormalError(::name(library) + " is not linked to pugs. Cannot use it!");
      // LCOV_EXCL_STOP
    }
  }

  static void
  checkBuiltinMethod(const LSMethod method)
  {
    switch (method) {
    case LSMethod::cg:
    case LSMethod::bicgstab: {
      break;
    }
    default: {
      throw NormalError(name(method) + " is not a builtin linear solver!");
    }
    }
  }

  static void
  checkPETScMethod(const LSMethod method)
  {
    switch (method) {
    case LSMethod::cg:
    case LSMethod::bicgstab:
    case LSMethod::bicgstab2:
    case LSMethod::gmres:
    case LSMethod::lu:
    case LSMethod::cholesky: {
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw NormalError(name(method) + " is not a builtin linear solver!");
    }
      // LCOV_EXCL_STOP
    }
  }

  static void
  checkBuiltinPrecond(const LSPrecond precond)
  {
    switch (precond) {
    case LSPrecond::none: {
      break;
    }
    default: {
      throw NormalError(name(precond) + " is not a builtin preconditioner!");
    }
    }
  }

  static void
  checkPETScPrecond(const LSPrecond precond)
  {
    switch (precond) {
    case LSPrecond::none:
    case LSPrecond::amg:
    case LSPrecond::diagonal:
    case LSPrecond::incomplete_cholesky:
    case LSPrecond::incomplete_LU: {
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw NormalError(name(precond) + " is not a PETSc preconditioner!");
    }
      // LCOV_EXCL_STOP
    }
  }

  static void
  checkOptions(const LinearSolverOptions& options)
  {
    switch (options.library()) {
    case LSLibrary::builtin: {
      checkBuiltinMethod(options.method());
      checkBuiltinPrecond(options.precond());
      break;
    }
    case LSLibrary::petsc: {
      checkPETScMethod(options.method());
      checkPETScPrecond(options.precond());
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("undefined options compatibility for this library (" + ::name(options.library()) + ")!");
    }
      // LCOV_EXCL_STOP
    }
  }

  template <typename MatrixType, typename SolutionVectorType, typename RHSVectorType>
  static void
  builtinSolveLocalSystem(const MatrixType& A,
                          SolutionVectorType& x,
                          const RHSVectorType& b,
                          const LinearSolverOptions& options)
  {
    if (options.precond() != LSPrecond::none) {
      // LCOV_EXCL_START
      throw UnexpectedError("builtin linear solver do not allow any preconditioner!");
      // LCOV_EXCL_STOP
    }
    switch (options.method()) {
    case LSMethod::cg: {
      CG{A, x, b, options.epsilon(), options.maximumIteration(), options.verbose()};
      break;
    }
    case LSMethod::bicgstab: {
      BiCGStab{A, x, b, options.epsilon(), options.maximumIteration(), options.verbose()};
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw NotImplementedError("undefined builtin method: " + name(options.method()));
    }
      // LCOV_EXCL_STOP
    }
  }

#ifdef PUGS_HAS_PETSC
  static int
  petscMonitor(KSP, int i, double residu, void*)
  {
    std::cout << "  - iteration: " << std::setw(6) << i << " residu: " << std::scientific << residu << '\n';
    return 0;
  }

  template <typename MatrixType, typename SolutionVectorType, typename RHSVectorType>
  static void
  petscSolveLocalSystem(const MatrixType& A,
                        SolutionVectorType& x,
                        const RHSVectorType& b,
                        const LinearSolverOptions& options)
  {
    Assert((x.size() == b.size()) and (x.size() - A.numberOfColumns() == 0) and A.isSquare());

    Vec petscB;
    VecCreateMPIWithArray(PETSC_COMM_SELF, 1, b.size(), b.size(), &b[0], &petscB);
    Vec petscX;
    VecCreateMPIWithArray(PETSC_COMM_SELF, 1, x.size(), x.size(), &x[0], &petscX);

    PETScAijMatrixEmbedder petscA(A);

    KSP ksp;
    KSPCreate(PETSC_COMM_SELF, &ksp);
    KSPSetTolerances(ksp, options.epsilon(), options.epsilon(), 1E5, options.maximumIteration());

    KSPSetOperators(ksp, petscA, petscA);

    PC pc;
    KSPGetPC(ksp, &pc);

    bool direct_solver = false;

    switch (options.method()) {
    case LSMethod::bicgstab: {
      KSPSetType(ksp, KSPBCGS);
      break;
    }
    case LSMethod::bicgstab2: {
      KSPSetType(ksp, KSPBCGSL);
      KSPBCGSLSetEll(ksp, 2);
      break;
    }
    case LSMethod::cg: {
      KSPSetType(ksp, KSPCG);
      break;
    }
    case LSMethod::gmres: {
      KSPSetType(ksp, KSPGMRES);

      break;
    }
    case LSMethod::lu: {
      KSPSetType(ksp, KSPPREONLY);
      PCSetType(pc, PCLU);
      PCFactorSetShiftType(pc, MAT_SHIFT_NONZERO);
      direct_solver = true;
      break;
    }
    case LSMethod::cholesky: {
      KSPSetType(ksp, KSPPREONLY);
      PCSetType(pc, PCCHOLESKY);
      PCFactorSetShiftType(pc, MAT_SHIFT_NONZERO);
      direct_solver = true;
      break;
    }
      // LCOV_EXCL_START
    default: {
      throw UnexpectedError("unexpected method: " + name(options.method()));
    }
      // LCOV_EXCL_STOP
    }

    if (not direct_solver) {
      switch (options.precond()) {
      case LSPrecond::amg: {
        PCSetType(pc, PCGAMG);
        break;
      }
      case LSPrecond::diagonal: {
        PCSetType(pc, PCJACOBI);
        break;
      }
      case LSPrecond::incomplete_LU: {
        PCSetType(pc, PCILU);
        break;
      }
      case LSPrecond::incomplete_cholesky: {
        PCSetType(pc, PCICC);
        break;
      }
      case LSPrecond::none: {
        PCSetType(pc, PCNONE);
        break;
      }
        // LCOV_EXCL_START
      default: {
        throw UnexpectedError("unexpected preconditioner: " + name(options.precond()));
      }
        // LCOV_EXCL_STOP
      }
    }
    if (options.verbose()) {
      KSPMonitorSet(ksp, petscMonitor, 0, 0);
    }

    KSPSolve(ksp, petscB, petscX);

    VecDestroy(&petscB);
    VecDestroy(&petscX);
    KSPDestroy(&ksp);
  }

#else   // PUGS_HAS_PETSC

  // LCOV_EXCL_START
  template <typename MatrixType, typename SolutionVectorType, typename RHSVectorType>
  static void
  petscSolveLocalSystem(const MatrixType&, SolutionVectorType&, const RHSVectorType&, const LinearSolverOptions&)
  {
    checkHasLibrary(LSLibrary::petsc);
    throw UnexpectedError("unexpected situation should not reach this point!");
  }
  // LCOV_EXCL_STOP

#endif   // PUGS_HAS_PETSC

  template <typename MatrixType, typename SolutionVectorType, typename RHSVectorType>
  static void
  solveLocalSystem(const MatrixType& A,
                   SolutionVectorType& x,
                   const RHSVectorType& b,
                   const LinearSolverOptions& options)
  {
    switch (options.library()) {
    case LSLibrary::builtin: {
      builtinSolveLocalSystem(A, x, b, options);
      break;
    }
      // LCOV_EXCL_START
    case LSLibrary::petsc: {
      // not covered since if PETSc is not linked this point is
      // unreachable: LinearSolver throws an exception at construction
      // in this case.
      petscSolveLocalSystem(A, x, b, options);
      break;
    }
    default: {
      throw UnexpectedError(::name(options.library()) + " cannot solve local systems");
    }
      // LCOV_EXCL_STOP
    }
  }
};

bool
LinearSolver::hasLibrary(LSLibrary library) const
{
  return Internals::hasLibrary(library);
}

void
LinearSolver::checkOptions(const LinearSolverOptions& options) const
{
  Internals::checkOptions(options);
}

void
LinearSolver::solveLocalSystem(const CRSMatrix<double, int>& A, Vector<double>& x, const Vector<const double>& b)
{
  Internals::solveLocalSystem(A, x, b, m_options);
}

void
LinearSolver::solveLocalSystem(const SmallMatrix<double>& A, SmallVector<double>& x, const SmallVector<const double>& b)
{
  Internals::solveLocalSystem(A, x, b, m_options);
}

LinearSolver::LinearSolver(const LinearSolverOptions& options) : m_options{options}
{
  Internals::checkHasLibrary(m_options.library());
  Internals::checkOptions(options);
}

LinearSolver::LinearSolver() : LinearSolver{LinearSolverOptions::default_options} {}
