#ifndef MESSENGER_HPP
#define MESSENGER_HPP

#include <PastisMacros.hpp>
#include <PastisAssert.hpp>

#include <Array.hpp>

#include <type_traits>

#include <pastis_config.hpp>
#ifdef PASTIS_HAS_MPI
#include <mpi.h>
#endif // PASTIS_HAS_MPI

#warning REMOVE
enum class CellType : unsigned short;

class Messenger
{
 private:
#ifdef PASTIS_HAS_MPI
  struct Helper
  {
    template<typename DataType>
    static PASTIS_INLINE
    MPI_Datatype mpiType()
    {
      if constexpr (std::is_const_v<DataType>) {
        return mpiType<std::remove_const_t<DataType>>();
      } else {
        static_assert(std::is_arithmetic_v<DataType>,
                      "Unexpected arithmetic type! Should not occur!");
        static_assert(not std::is_arithmetic_v<DataType>,
                      "MPI_Datatype are only defined for arithmetic types!");
        return MPI_Datatype();
      }
    }
  };
#endif PASTIS_HAS_MPI

  static Messenger* m_instance;
  Messenger(int& argc, char* argv[]);

  int m_rank{0};
  int m_size{1};

  Array<int> _allGather(int& data) const;

  int _broadcast(int& data, int root_rank) const;
  Array<int> _broadcast(Array<int>& array, int root_rank) const;
  Array<int> _allToAll(const Array<int>& sent_array, Array<int>& recv_array) const;

  template <typename DataType>
  void _exchange(const std::vector<Array<DataType>>& sent_array_list,
                 std::vector<Array<std::remove_const_t<DataType>>>& recv_array_list) const
  {
#ifdef PASTIS_HAS_MPI
    std::vector<MPI_Request> request_list;

#warning clean-up
    MPI_Datatype type = [&] () -> MPI_Datatype {
                          if constexpr (std::is_same_v<CellType,std::remove_const_t<DataType>>) {
                            return MPI_SHORT;
                          } else {
                            return Messenger::Helper::mpiType<DataType>();
                          }
                        } ();


    for (size_t i_send=0; i_send<sent_array_list.size(); ++i_send) {
      const Array<DataType> sent_array = sent_array_list[i_send];
      if (sent_array.size()>0) {
        MPI_Request request;
        MPI_Isend(&(sent_array[0]), sent_array.size(), type, i_send, 0, MPI_COMM_WORLD, &request);
        request_list.push_back(request);
      }
    }

    for (size_t i_recv=0; i_recv<recv_array_list.size(); ++i_recv) {
      Array<std::remove_const_t<DataType>> recv_array = recv_array_list[i_recv];
      if (recv_array.size()>0) {
        MPI_Request request;
        MPI_Irecv(&(recv_array[0]), recv_array.size(), type, i_recv, 0, MPI_COMM_WORLD, &request);
        request_list.push_back(request);
      }
    }

    std::vector<MPI_Status> status_list(request_list.size());
    if (MPI_SUCCESS != MPI_Waitall(request_list.size(), &(request_list[0]), &(status_list[0]))) {
      std::cerr << "Communication error!\n";
      std::exit(1);
    }

#else // PASTIS_HAS_MPI
    std::cerr << "NIY\n";
    std::exit(1);
#endif // PASTIS_HAS_MPI
  }


 public:
  static void create(int& argc, char* argv[]);
  static void destroy();

  PASTIS_INLINE
  static Messenger& getInstance()
  {
    Assert(m_instance != nullptr);
    return *m_instance;
  }

  PASTIS_INLINE
  const int& rank() const
  {
    return m_rank;
  }

  PASTIS_INLINE
  const int& size() const
  {
    return m_size;
  }

  void barrier() const;

  template <typename DataType>
  PASTIS_INLINE
  Array<DataType> allGather(const DataType& data) const
  {
    if constexpr(std::is_same<DataType, int>()) {
      int int_data = data;
      return _allGather(int_data);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename DataType>
  PASTIS_INLINE
  Array<DataType> allToAll(const Array<DataType>& sent_array) const
  {
    Assert(sent_array.size() == static_cast<size_t>(m_size));
    if constexpr(std::is_same<DataType, int>()) {
      Array<int> recv_array(m_size);
      return _allToAll(sent_array, recv_array);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename DataType>
  PASTIS_INLINE
  DataType broadcast(const DataType& data, int root_rank) const
  {
    if constexpr(std::is_same<DataType, int>()) {
      int int_data = data;
      return _broadcast(int_data, root_rank);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename DataType>
  PASTIS_INLINE
  Array<DataType> broadcast(const Array<DataType>& array, int root_rank) const
  {
    if constexpr(std::is_same<DataType, int>()) {
      Array<int> int_array = array;
      return _broadcast(int_array, root_rank);
    } else {
      static_assert(std::is_same<DataType, int>(), "unexpected type of data");
    }
  }

  template <typename SendDataType,
            typename RecvDataType>
  PASTIS_INLINE
  void exchange(const std::vector<Array<SendDataType>>& sent_array_list,
                std::vector<Array<RecvDataType>>& recv_array_list) const
  {
    static_assert(std::is_same_v<std::remove_const_t<SendDataType>,RecvDataType>,
                  "send and receive data type do not match");
    static_assert(not std::is_const_v<RecvDataType>,
                  "receive data type cannot be const");

    if constexpr(std::is_arithmetic_v<RecvDataType>) {
      _exchange(sent_array_list, recv_array_list);
    } else if constexpr(std::is_same<RecvDataType, CellType>()) {
      _exchange(sent_array_list, recv_array_list);
    } else {
      static_assert(std::is_same<RecvDataType, int>(), "unexpected type of data");
    }
  }

  Messenger(const Messenger&) = delete;
  ~Messenger();
};

PASTIS_INLINE
const Messenger& messenger()
{
  return Messenger::getInstance();
}

[[deprecated("use better name")]]
PASTIS_INLINE
const int& commRank()
{
  return messenger().rank();
}

[[deprecated("use better name")]]
PASTIS_INLINE
const int& commSize()
{
  return messenger().size();
}

PASTIS_INLINE
void barrier()
{
  return messenger().barrier();
}

template <typename DataType>
PASTIS_INLINE
DataType broadcast(const DataType& data, int root_rank)
{
  return messenger().broadcast(data, root_rank);
}

template <typename DataType>
PASTIS_INLINE
Array<DataType> allGather(const DataType& data)
{
  return messenger().allGather(data);
}

template <typename DataType>
PASTIS_INLINE
Array<DataType> allToAll(const Array<DataType>& array)
{
  return messenger().allToAll(array);
}

template <typename DataType>
PASTIS_INLINE
Array<DataType> broadcast(const Array<DataType>& array, int root_rank)
{
  return messenger().broadcast(array, root_rank);
}

template <typename SendDataType,
          typename RecvDataType>
PASTIS_INLINE
void exchange(const std::vector<Array<SendDataType>>& sent_array_list,
              std::vector<Array<RecvDataType>>& recv_array_list)
{
  static_assert(std::is_same_v<std::remove_const_t<SendDataType>,RecvDataType>,
                "send and receive data type do not match");
  static_assert(not std::is_const_v<RecvDataType>,
                "receive data type cannot be const");

  messenger().exchange(sent_array_list, recv_array_list);
}

#ifdef PASTIS_HAS_MPI

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<char>() {return MPI_CHAR; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<int8_t>() {return MPI_INT8_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<int16_t>() {return MPI_INT16_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<int32_t>() {return MPI_INT32_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<int64_t>() {return MPI_INT64_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<uint8_t>() {return MPI_UINT8_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<uint16_t>() {return MPI_UINT16_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<uint32_t>() {return MPI_UINT32_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<uint64_t>() {return MPI_UINT64_T; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<signed long long int>() {return MPI_LONG_LONG_INT; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<unsigned long long int>() {return MPI_UNSIGNED_LONG_LONG; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<float>() {return MPI_FLOAT; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<double>() {return MPI_DOUBLE; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<long double>() {return MPI_LONG_DOUBLE; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<wchar_t>() {return MPI_WCHAR; }

template<> PASTIS_INLINE MPI_Datatype
Messenger::Helper::mpiType<bool>() {return MPI_CXX_BOOL; }

#endif // PASTIS_HAS_MPI

#endif // MESSENGER_HPP
