#ifndef VTK_WRITER_HPP
#define VTK_WRITER_HPP

#include <string>
#include <fstream>
#include <iomanip>
#include <sstream>
#include <TinyVector.hpp>
#include <IConnectivity.hpp>

#include <ItemValue.hpp>

class VTKWriter
{
 private:
  const std::string m_base_filename;
  unsigned int m_file_number;
  double m_last_time;
  const double m_time_period;

 public:
  template <typename MeshType>
  void write(const MeshType& mesh,
             const double& time,
             const bool& forced_output = false)
  {
    if (time == m_last_time) return; // output already performed
    if ((time - m_last_time >= m_time_period) or forced_output) {
      m_last_time = time;
    } else {
      return;
    }
    std::ostringstream sout;
    sout << m_base_filename << '.' << std::setfill('0') << std::setw(4) << m_file_number << ".vtu" << std::ends;
    std::ofstream fout(sout.str());
    fout << "<?xml version=\"1.0\"?>\n";
    fout << "<VTKFile type=\"UnstructuredGrid\">\n";
    fout << "<UnstructuredGrid>\n";
    fout << "<Piece NumberOfPoints=\""<< mesh.numberOfNodes()
         << "\" NumberOfCells=\"" << mesh.numberOfCells() << "\">\n";

    fout << "<Points>\n";
    fout << "<DataArray Name=\"Positions\" NumberOfComponents=\"3\" type=\"Float64\" format=\"ascii\">\n";
    using Rd = TinyVector<MeshType::dimension>;
    const NodeValue<const Rd>& xr = mesh.xr();
    if constexpr(MeshType::dimension == 1) {
      for (NodeId r=0; r<mesh.numberOfNodes(); ++r) {
        for (unsigned short i=0; i<1; ++i) {
          fout << xr[r][i] << ' ';
        }
        fout << "0 0 "; // VTK requires 3 components
      }
    } else if constexpr (MeshType::dimension == 2) {
      for (NodeId r=0; r<mesh.numberOfNodes(); ++r) {
        for (unsigned short i=0; i<2; ++i) {
          fout << xr[r][i] << ' ';
        }
        fout << "0 "; // VTK requires 3 components
      }
    } else {
      for (NodeId r=0; r<mesh.numberOfNodes(); ++r) {
        for (unsigned short i=0; i<3; ++i) {
          fout << xr[r][i] << ' ';
        }
      }
    }
    fout << '\n';
    fout << "</DataArray>\n";
    fout << "</Points>\n";

    fout << "<Cells>\n";

    fout << "<DataArray type=\"Int32\" Name=\"connectivity\" NumberOfComponents=\"1\" format=\"ascii\">\n";

    const auto& cell_to_node_matrix
        = mesh.connectivity().cellToNodeMatrix();

    for (CellId j=0; j<mesh.numberOfCells(); ++j) {
      const auto& cell_nodes = cell_to_node_matrix[j];
      for (unsigned short r=0; r<cell_nodes.size(); ++r) {
        fout << cell_nodes[r] << ' ';
      }
    }
    fout << '\n';
    fout << "</DataArray>\n";

    fout << "<DataArray type=\"UInt32\" Name=\"offsets\" NumberOfComponents=\"1\" format=\"ascii\">\n";
    {
      unsigned int offset=0;
      for (CellId j=0; j<mesh.numberOfCells(); ++j) {
        const auto& cell_nodes = cell_to_node_matrix[j];
        offset += cell_nodes.size();
        fout << offset << ' ';
      }
    }
    fout << '\n';
    fout << "</DataArray>\n";

    fout << "<DataArray type=\"Int8\" Name=\"types\" NumberOfComponents=\"1\" format=\"ascii\">\n";
    for (CellId j=0; j<mesh.numberOfCells(); ++j) {
      const auto& cell_nodes = cell_to_node_matrix[j];
      switch (cell_nodes.size()) {
        case 2: {
          fout << "3 ";
          break;
        }
        case 3: {
          fout << "5 ";
          break;
        }
        case 4: {
          if (mesh.meshDimension() == 3) {
            fout << "10 ";
          } else {
            fout << "9 ";
          }
          break;
        }
        case 8: {
          fout << "12 ";
          break;
        }
        default: {
          fout << "7 ";
          break;
        }
      }
    }
    fout << '\n';
    fout << "</DataArray>\n";

    fout << "</Cells>\n";
    fout << "</Piece>\n";
    fout << "</UnstructuredGrid>\n";
    fout << "</VTKFile>\n";

    m_file_number++;
  }
  VTKWriter(const std::string& base_filename,
            const double time_period)
      : m_base_filename(base_filename),
        m_file_number  (0),
        m_last_time    (-std::numeric_limits<double>::max()),
        m_time_period  (time_period)
  {}

  ~VTKWriter() = default;
};

#endif // VTK_WRITER_HPP
