#ifndef SUBITEM_VALUE_PER_ITEM_HPP
#define SUBITEM_VALUE_PER_ITEM_HPP

#include <Kokkos_StaticCrsGraph.hpp>

#include <PastisAssert.hpp>

#include <Array.hpp>
#include <ItemType.hpp>

#include <ItemId.hpp>

#include <ConnectivityMatrix.hpp>
#include <IConnectivity.hpp>

template <typename DataType,
          ItemType sub_item_type,
          ItemType item_type,
          typename Allowed=void>
class SubItemValuePerItem;

template <typename DataType,
          ItemType sub_item_type,
          ItemType item_type>
class SubItemValuePerItem<DataType,
                          sub_item_type,
                          item_type,
                          std::enable_if_t<sub_item_type != item_type>>
{
 public:
  static const ItemType item_t{item_type};
  static const ItemType sub_item_t{sub_item_type};

  using data_type = DataType;
  using ItemId = ItemIdT<item_type>;
  using index_type = ItemId;

 private:
  bool m_is_built{false};

  typename ConnectivityMatrix::HostRowType m_host_row_map;
  Array<DataType> m_values;

  // Allows const version to access our data
  friend SubItemValuePerItem<std::add_const_t<DataType>,
                             sub_item_type,
                             item_type>;

 public:
  class SubView
  {
   public:
    using data_type = DataType;

   private:
    PASTIS_RESTRICT DataType* const m_sub_values;
    const size_t m_size;

   public:
    PASTIS_INLINE
    const DataType& operator[](const size_t& i) const
    {
      Assert(i<m_size);
      return m_sub_values[i];
    }

    PASTIS_FORCEINLINE
    DataType& operator[](const size_t& i)
    {
      Assert(i<m_size);
      return m_sub_values[i];
    }

    PASTIS_INLINE
    const size_t& size() const
    {
      return m_size;
    }

    SubView(const SubView&) = delete;

    PASTIS_INLINE
    SubView(SubView&&) = default;

    PASTIS_INLINE
    SubView(const Array<DataType>& values,
            const size_t& begin,
            const size_t& end)
        : m_sub_values(&(values[begin])),
          m_size(end-begin)
    {
      Assert(begin<=end);
      Assert(end<=values.size());
    }
  };

  PASTIS_FORCEINLINE
  const bool& isBuilt() const
  {
    return m_is_built;
  }

  // Following Kokkos logic, these classes are view and const view does allow
  // changes in data
  PASTIS_FORCEINLINE
  DataType& operator()(const ItemId& j, const size_t& r) const
  {
    Assert(m_is_built);
    return m_values[m_host_row_map(size_t{j})+r];
  }

  template <typename IndexType>
  PASTIS_FORCEINLINE
  DataType& operator()(const IndexType& j, const size_t& r) const
  {
    static_assert(std::is_same<IndexType, size_t>(),
                  "SubItemValuePerItem indexed by ItemId");
    return m_values[m_host_row_map(size_t{j})+r];
  }

  PASTIS_INLINE
  size_t numberOfValues() const
  {
    Assert(m_is_built);
    return m_values.size();
  }

  // Following Kokkos logic, these classes are view and const view does allow
  // changes in data
  PASTIS_FORCEINLINE
  DataType& operator[](const size_t& i) const
  {
    Assert(m_is_built);
    return m_values[i];
  }

  template <typename IndexType>
  DataType& operator[](const IndexType& i) const
  {
    static_assert(std::is_same<IndexType, size_t>(),
                  "Access to SubItemValuePerItem's array must be indexed by size_t");
    return m_values[i];
  }

  PASTIS_INLINE
  size_t numberOfItems() const
  {
    Assert(m_is_built);
    Assert(m_host_row_map.extent(0) != 0);
    return m_host_row_map.extent(0);
  }

  PASTIS_INLINE
  size_t numberOfSubValues(const size_t& i_cell) const
  {
    Assert(m_is_built);
    return m_host_row_map(i_cell+1)-m_host_row_map(i_cell);
  }

  PASTIS_INLINE
  SubView itemValues(const size_t& i_cell)
  {
    Assert(m_is_built);
    const auto& cell_begin = m_host_row_map(i_cell);
    const auto& cell_end = m_host_row_map(i_cell+1);
    return SubView(m_values, cell_begin, cell_end);
  }

  // Following Kokkos logic, these classes are view and const view does allow
  // changes in data
  PASTIS_INLINE
  SubView itemValues(const size_t& i_cell) const
  {
    Assert(m_is_built);
    const auto& cell_begin = m_host_row_map(i_cell);
    const auto& cell_end = m_host_row_map(i_cell+1);
    return SubView(m_values, cell_begin, cell_end);
  }

  template <typename DataType2>
  PASTIS_INLINE
  SubItemValuePerItem&
  operator=(const SubItemValuePerItem<DataType2, sub_item_type, item_type>& sub_item_value_per_item)
  {
    // ensures that DataType is the same as source DataType2
    static_assert(std::is_same<std::remove_const_t<DataType>, std::remove_const_t<DataType2>>(),
                  "Cannot assign SubItemValuePerItem of different type");
    // ensures that const is not lost through copy
    static_assert(((std::is_const<DataType2>() and std::is_const<DataType>())
                   or not std::is_const<DataType2>()),
                  "Cannot assign SubItemValuePerItem of const data to SubItemValuePerItem of non-const data");
    m_host_row_map = sub_item_value_per_item.m_host_row_map;
    m_values = sub_item_value_per_item.m_values;

    m_is_built = sub_item_value_per_item.m_is_built;

    return *this;
  }

  template <typename DataType2>
  PASTIS_INLINE
  SubItemValuePerItem(const SubItemValuePerItem<DataType2, sub_item_type, item_type>& sub_item_value_per_item)
  {
    this->operator=(sub_item_value_per_item);
  }

  SubItemValuePerItem() = default;

  SubItemValuePerItem(const IConnectivity& connectivity)
      : m_is_built{true}
  {
    static_assert(not std::is_const<DataType>(),
                  "Cannot allocate SubItemValuePerItem of const data: only view is supported"); ;

    ConnectivityMatrix connectivity_matrix
        = connectivity._getMatrix(item_type, sub_item_type);

    m_host_row_map = connectivity_matrix.rowsMap();
    m_values = Array<std::remove_const_t<DataType>>(connectivity_matrix.numEntries());
  }

  ~SubItemValuePerItem() = default;
};

// Item values at nodes

template <typename DataType>
using NodeValuePerEdge = SubItemValuePerItem<DataType, ItemType::node, ItemType::edge>;

template <typename DataType>
using NodeValuePerFace = SubItemValuePerItem<DataType, ItemType::node, ItemType::face>;

template <typename DataType>
using NodeValuePerCell = SubItemValuePerItem<DataType, ItemType::node, ItemType::cell>;

// Item values at edges

template <typename DataType>
using EdgeValuePerNode = SubItemValuePerItem<DataType, ItemType::edge, ItemType::node>;

template <typename DataType>
using EdgeValuePerFace = SubItemValuePerItem<DataType, ItemType::edge, ItemType::face>;

template <typename DataType>
using EdgeValuePerCell = SubItemValuePerItem<DataType, ItemType::edge, ItemType::cell>;

// Item values at faces

template <typename DataType>
using FaceValuePerNode = SubItemValuePerItem<DataType, ItemType::face, ItemType::node>;

template <typename DataType>
using FaceValuePerEdge = SubItemValuePerItem<DataType, ItemType::face, ItemType::edge>;

template <typename DataType>
using FaceValuePerCell = SubItemValuePerItem<DataType, ItemType::face, ItemType::cell>;

// Item values at cells

template <typename DataType>
using CellValuePerNode = SubItemValuePerItem<DataType, ItemType::cell, ItemType::node>;

template <typename DataType>
using CellValuePerEdge = SubItemValuePerItem<DataType, ItemType::cell, ItemType::edge>;

template <typename DataType>
using CellValuePerFace = SubItemValuePerItem<DataType, ItemType::cell, ItemType::face>;

#endif // SUBITEM_VALUE_PER_ITEM_HPP
