#ifndef ITEM_TO_ITEM_MATRIX_HPP
#define ITEM_TO_ITEM_MATRIX_HPP

#include <ItemType.hpp>
#include <ItemId.hpp>

#include <PastisUtils.hpp>
#include <ConnectivityMatrix.hpp>

template <ItemType SourceItemType,
          ItemType TargetItemType>
class ItemToItemMatrix
{
 public:
  using SourceItemId = ItemIdT<SourceItemType>;
  using TargetItemId = ItemIdT<TargetItemType>;

  template <typename RowType>
  class SubItemList
  {
   private:
    const RowType m_row;

   public:

    PASTIS_INLINE
    size_t size() const
    {
      return m_row.length;
    }

    PASTIS_INLINE
    TargetItemId operator[](const size_t& j) const
    {
      return m_row(j);
    }

    PASTIS_INLINE
    SubItemList(const RowType& row)
        : m_row{row}
    {
      ;
    }

    PASTIS_INLINE
    SubItemList& operator=(const SubItemList&) = default;

    PASTIS_INLINE
    SubItemList& operator=(SubItemList&&) = default;

    PASTIS_INLINE
    SubItemList(const SubItemList&) = default;

    PASTIS_INLINE
    SubItemList(SubItemList&&) = default;

    PASTIS_INLINE
    ~SubItemList() = default;
  };

 private:
  const ConnectivityMatrix& m_connectivity_matrix;

 public:
  PASTIS_INLINE
  const auto operator[](const SourceItemId& source_id) const
  {
    using RowType = decltype(m_connectivity_matrix.rowConst(source_id));
    return SubItemList<RowType>(m_connectivity_matrix.rowConst(source_id));
  }

  template <typename IndexType>
  PASTIS_INLINE
  const auto operator[](const IndexType& source_id) const
  {
    static_assert(std::is_same<IndexType, SourceItemId>(),
                  "ItemToItemMatrix must be indexed using correct ItemId");
    using RowType = decltype(m_connectivity_matrix.rowConst(source_id));
    return SubItemList<RowType>(m_connectivity_matrix.rowConst(source_id));
  }

  PASTIS_INLINE
  ItemToItemMatrix(const ConnectivityMatrix& connectivity_matrix)
      : m_connectivity_matrix{connectivity_matrix}
  {
    ;
  }

  PASTIS_INLINE
  ItemToItemMatrix& operator=(const ItemToItemMatrix&) = default;

  PASTIS_INLINE
  ItemToItemMatrix& operator=(ItemToItemMatrix&&) = default;

  PASTIS_INLINE
  ItemToItemMatrix(ItemToItemMatrix&&) = default;

  PASTIS_INLINE
  ItemToItemMatrix(const ItemToItemMatrix&) = default;

  PASTIS_INLINE
  ~ItemToItemMatrix() = default;
};

#endif // ITEM_TO_ITEM_MATRIX_HPP
