#include <language/algorithms/UnsteadyElasticity.hpp>

#include <algebra/CRSMatrix.hpp>
#include <algebra/CRSMatrixDescriptor.hpp>
#include <algebra/LeastSquareSolver.hpp>
#include <algebra/LinearSolver.hpp>
#include <algebra/SmallMatrix.hpp>
#include <algebra/TinyVector.hpp>
#include <algebra/Vector.hpp>
#include <language/utils/InterpolateItemValue.hpp>
#include <mesh/Connectivity.hpp>
#include <mesh/DualConnectivityManager.hpp>
#include <mesh/DualMeshManager.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/MeshData.hpp>
#include <mesh/MeshDataManager.hpp>
#include <mesh/MeshFaceBoundary.hpp>
#include <mesh/PrimalToDiamondDualConnectivityDataMapper.hpp>
#include <scheme/DirichletBoundaryConditionDescriptor.hpp>
#include <scheme/NeumannBoundaryConditionDescriptor.hpp>
#include <scheme/SymmetryBoundaryConditionDescriptor.hpp>
#include <scheme/VectorDiamondScheme.hpp>

template <size_t Dimension>
UnsteadyElasticity<Dimension>::UnsteadyElasticity(
  std::shared_ptr<const IMesh> i_mesh,
  const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list,
  const FunctionSymbolId& lambda_id,
  const FunctionSymbolId& mu_id,
  const FunctionSymbolId& f_id,
  const FunctionSymbolId& U_id,
  const FunctionSymbolId& U_init_id,
  const double& Tf,
  const double& dt)
{
  using ConnectivityType = Connectivity<Dimension>;
  using MeshType         = Mesh<ConnectivityType>;
  using MeshDataType     = MeshData<Dimension>;

  using BoundaryCondition =
    std::variant<DirichletBoundaryCondition, NormalStrainBoundaryCondition, SymmetryBoundaryCondition>;

  using BoundaryConditionList = std::vector<BoundaryCondition>;

  BoundaryConditionList boundary_condition_list;

  std::cout << "number of bc descr = " << bc_descriptor_list.size() << '\n';
  std::shared_ptr mesh = std::dynamic_pointer_cast<const MeshType>(i_mesh);

  NodeValue<bool> is_dirichlet{mesh->connectivity()};
  is_dirichlet.fill(false);
  NodeValue<TinyVector<Dimension>> dirichlet_value{mesh->connectivity()};
  {
    TinyVector<Dimension> nan_tiny_vector;
    for (size_t i = 0; i < Dimension; ++i) {
      nan_tiny_vector[i] = std::numeric_limits<double>::signaling_NaN();
    }
    dirichlet_value.fill(nan_tiny_vector);
  }

  for (const auto& bc_descriptor : bc_descriptor_list) {
    bool is_valid_boundary_condition = true;

    switch (bc_descriptor->type()) {
    case IBoundaryConditionDescriptor::Type::symmetry: {
      const SymmetryBoundaryConditionDescriptor& sym_bc_descriptor =
        dynamic_cast<const SymmetryBoundaryConditionDescriptor&>(*bc_descriptor);
      if constexpr (Dimension > 1) {
        MeshFaceBoundary mesh_face_boundary = getMeshFaceBoundary(*mesh, sym_bc_descriptor.boundaryDescriptor());
        boundary_condition_list.push_back(SymmetryBoundaryCondition{mesh_face_boundary.faceList()});

      } else {
        throw NotImplementedError("Symmetry conditions are not supported in 1d");
      }

      break;
    }
    case IBoundaryConditionDescriptor::Type::dirichlet: {
      const DirichletBoundaryConditionDescriptor& dirichlet_bc_descriptor =
        dynamic_cast<const DirichletBoundaryConditionDescriptor&>(*bc_descriptor);
      if (dirichlet_bc_descriptor.name() == "dirichlet") {
        if constexpr (Dimension > 1) {
          MeshFaceBoundary mesh_face_boundary =
            getMeshFaceBoundary(*mesh, dirichlet_bc_descriptor.boundaryDescriptor());
          MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(*mesh);

          const FunctionSymbolId g_id = dirichlet_bc_descriptor.rhsSymbolId();

          Array<const TinyVector<Dimension>> value_list = InterpolateItemValue<TinyVector<Dimension>(
            TinyVector<Dimension>)>::template interpolate<ItemType::face>(g_id, mesh_data.xl(),
                                                                          mesh_face_boundary.faceList());

          boundary_condition_list.push_back(DirichletBoundaryCondition{mesh_face_boundary.faceList(), value_list});
        } else {
          throw NotImplementedError("Dirichlet conditions are not supported in 1d");
        }
      } else if (dirichlet_bc_descriptor.name() == "normal_strain") {
        if constexpr (Dimension > 1) {
          MeshFaceBoundary mesh_face_boundary =
            getMeshFaceBoundary(*mesh, dirichlet_bc_descriptor.boundaryDescriptor());
          MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(*mesh);

          const FunctionSymbolId g_id = dirichlet_bc_descriptor.rhsSymbolId();

          Array<const TinyVector<Dimension>> value_list = InterpolateItemValue<TinyVector<Dimension>(
            TinyVector<Dimension>)>::template interpolate<ItemType::face>(g_id, mesh_data.xl(),
                                                                          mesh_face_boundary.faceList());
          boundary_condition_list.push_back(NormalStrainBoundaryCondition{mesh_face_boundary.faceList(), value_list});
        } else {
          throw NotImplementedError("Normal strain conditions are not supported in 1d");
        }

      } else {
        is_valid_boundary_condition = false;
      }
      break;
    }
    default: {
      is_valid_boundary_condition = false;
    }
    }
    if (not is_valid_boundary_condition) {
      std::ostringstream error_msg;
      error_msg << *bc_descriptor << " is an invalid boundary condition for elasticity equation";
      throw NormalError(error_msg.str());
    }
  }

  if constexpr (Dimension > 1) {
    const CellValue<const size_t> cell_dof_number = [&] {
      CellValue<size_t> compute_cell_dof_number{mesh->connectivity()};
      parallel_for(
        mesh->numberOfCells(), PUGS_LAMBDA(CellId cell_id) { compute_cell_dof_number[cell_id] = cell_id; });
      return compute_cell_dof_number;
    }();
    size_t number_of_dof = mesh->numberOfCells();

    const FaceValue<const size_t> face_dof_number = [&] {
      FaceValue<size_t> compute_face_dof_number{mesh->connectivity()};
      compute_face_dof_number.fill(std::numeric_limits<size_t>::max());
      for (const auto& boundary_condition : boundary_condition_list) {
        std::visit(
          [&](auto&& bc) {
            using T = std::decay_t<decltype(bc)>;
            if constexpr ((std::is_same_v<T, NormalStrainBoundaryCondition>) or
                          (std::is_same_v<T, SymmetryBoundaryCondition>) or
                          (std::is_same_v<T, DirichletBoundaryCondition>)) {
              const auto& face_list = bc.faceList();

              for (size_t i_face = 0; i_face < face_list.size(); ++i_face) {
                const FaceId face_id = face_list[i_face];
                if (compute_face_dof_number[face_id] != std::numeric_limits<size_t>::max()) {
                  std::ostringstream os;
                  os << "The face " << face_id << " is used at least twice for boundary conditions";
                  throw NormalError(os.str());
                } else {
                  compute_face_dof_number[face_id] = number_of_dof++;
                }
              }
            }
          },
          boundary_condition);
      }

      return compute_face_dof_number;
    }();

    const auto& primal_face_to_node_matrix             = mesh->connectivity().faceToNodeMatrix();
    const auto& face_to_cell_matrix                    = mesh->connectivity().faceToCellMatrix();
    const FaceValue<const bool> primal_face_is_neumann = [&] {
      FaceValue<bool> face_is_neumann{mesh->connectivity()};
      face_is_neumann.fill(false);
      for (const auto& boundary_condition : boundary_condition_list) {
        std::visit(
          [&](auto&& bc) {
            using T = std::decay_t<decltype(bc)>;
            if constexpr ((std::is_same_v<T, NormalStrainBoundaryCondition>)) {
              const auto& face_list = bc.faceList();

              for (size_t i_face = 0; i_face < face_list.size(); ++i_face) {
                const FaceId face_id     = face_list[i_face];
                face_is_neumann[face_id] = true;
              }
            }
          },
          boundary_condition);
      }

      return face_is_neumann;
    }();

    FaceValue<bool> primal_face_is_symmetry = [&] {
      FaceValue<bool> face_is_symmetry{mesh->connectivity()};
      face_is_symmetry.fill(false);
      for (const auto& boundary_condition : boundary_condition_list) {
        std::visit(
          [&](auto&& bc) {
            using T = std::decay_t<decltype(bc)>;
            if constexpr ((std::is_same_v<T, SymmetryBoundaryCondition>)) {
              const auto& face_list = bc.faceList();

              for (size_t i_face = 0; i_face < face_list.size(); ++i_face) {
                const FaceId face_id      = face_list[i_face];
                face_is_symmetry[face_id] = true;
              }
            }
          },
          boundary_condition);
      }

      return face_is_symmetry;
    }();

    NodeValue<bool> primal_node_is_on_boundary(mesh->connectivity());
    if (parallel::size() > 1) {
      throw NotImplementedError("Calculation of node_is_on_boundary is incorrect");
    }

    primal_node_is_on_boundary.fill(false);
    for (FaceId face_id = 0; face_id < mesh->numberOfFaces(); ++face_id) {
      if (face_to_cell_matrix[face_id].size() == 1) {
        for (size_t i_node = 0; i_node < primal_face_to_node_matrix[face_id].size(); ++i_node) {
          NodeId node_id                      = primal_face_to_node_matrix[face_id][i_node];
          primal_node_is_on_boundary[node_id] = true;
        }
      }
    }

    FaceValue<bool> primal_face_is_on_boundary(mesh->connectivity());
    if (parallel::size() > 1) {
      throw NotImplementedError("Calculation of face_is_on_boundary is incorrect");
    }

    primal_face_is_on_boundary.fill(false);
    for (FaceId face_id = 0; face_id < mesh->numberOfFaces(); ++face_id) {
      if (face_to_cell_matrix[face_id].size() == 1) {
        primal_face_is_on_boundary[face_id] = true;
      }
    }

    FaceValue<bool> primal_face_is_dirichlet(mesh->connectivity());
    if (parallel::size() > 1) {
      throw NotImplementedError("Calculation of face_is_neumann is incorrect");
    }

    primal_face_is_dirichlet.fill(false);
    for (FaceId face_id = 0; face_id < mesh->numberOfFaces(); ++face_id) {
      primal_face_is_dirichlet[face_id] = (primal_face_is_on_boundary[face_id] && (!primal_face_is_neumann[face_id]) &&
                                           (!primal_face_is_symmetry[face_id]));
    }
    MeshDataType& mesh_data         = MeshDataManager::instance().getMeshData(*mesh);
    const auto& node_to_cell_matrix = mesh->connectivity().nodeToCellMatrix();
    const auto& node_to_face_matrix = mesh->connectivity().nodeToFaceMatrix();

    {
      CellValue<TinyVector<Dimension>> velocity = InterpolateItemValue<TinyVector<Dimension>(
        TinyVector<Dimension>)>::template interpolate<ItemType::cell>(U_id, mesh_data.xj());
      CellValue<TinyVector<Dimension>> Uj       = InterpolateItemValue<TinyVector<Dimension>(
        TinyVector<Dimension>)>::template interpolate<ItemType::cell>(U_init_id, mesh_data.xj());

      CellValue<TinyVector<Dimension>> fj            = InterpolateItemValue<TinyVector<Dimension>(
        TinyVector<Dimension>)>::template interpolate<ItemType::cell>(f_id, mesh_data.xj());
      FaceValue<TinyVector<Dimension>> velocity_face = InterpolateItemValue<TinyVector<Dimension>(
        TinyVector<Dimension>)>::template interpolate<ItemType::face>(U_id, mesh_data.xl());
      FaceValue<TinyVector<Dimension>> Ul            = InterpolateItemValue<TinyVector<Dimension>(
        TinyVector<Dimension>)>::template interpolate<ItemType::face>(U_init_id, mesh_data.xl());

      double time = 0;
      Assert(dt > 0, "The time-step have to be positive!");
      const CellValue<const double> primal_Vj = mesh_data.Vj();

      InterpolationWeightsManager iwm(mesh, primal_face_is_on_boundary, primal_node_is_on_boundary,
                                      primal_face_is_symmetry);
      iwm.compute();
      CellValuePerNode<double> w_rj = iwm.wrj();
      FaceValuePerNode<double> w_rl = iwm.wrl();
      do {
        double deltat = std::min(dt, Tf - time);
        std::cout << "Current time = " << time << " time-step = " << deltat << " final time = " << Tf << "\n";
        LegacyVectorDiamondScheme<Dimension>(i_mesh, bc_descriptor_list, lambda_id, mu_id, f_id, Uj, Ul, Tf, deltat,
                                             w_rj, w_rl);
        time += deltat;
      } while (time < Tf && std::abs(time - Tf) > 1e-15);

      const FaceValue<const double> mes_l = [&] {
        if constexpr (Dimension == 1) {
          FaceValue<double> compute_mes_l{mesh->connectivity()};
          compute_mes_l.fill(1);
          return compute_mes_l;
        } else {
          return mesh_data.ll();
        }
      }();

      Vector<double> Uexacte{mesh->numberOfCells() * Dimension};
      for (CellId j = 0; j < mesh->numberOfCells(); ++j) {
        for (size_t l = 0; l < Dimension; ++l) {
          Uexacte[(cell_dof_number[j] * Dimension) + l] = velocity[j][l];
        }
      }

      Vector<double> error{mesh->numberOfCells() * Dimension};
      for (CellId cell_id = 0; cell_id < mesh->numberOfCells(); ++cell_id) {
        for (size_t i = 0; i < Dimension; ++i) {
          error[(cell_id * Dimension) + i] = (Uj[cell_id][i] - velocity[cell_id][i]) * sqrt(primal_Vj[cell_id]);
        }
      }
      Vector<double> error_face{mesh->numberOfFaces() * Dimension};
      parallel_for(
        mesh->numberOfFaces(), PUGS_LAMBDA(FaceId face_id) {
          if (primal_face_is_on_boundary[face_id]) {
            for (size_t i = 0; i < Dimension; ++i) {
              error_face[face_id * Dimension + i] = (Ul[face_id][i] - velocity_face[face_id][i]) * sqrt(mes_l[face_id]);
            }
          } else {
            error_face[face_id] = 0;
          }
        });

      std::cout << "||Error||_2 (cell)= " << std::sqrt(dot(error, error)) << "\n";
      std::cout << "||Error||_2 (face)= " << std::sqrt(dot(error_face, error_face)) << "\n";
      std::cout << "||Error||_2 (total)= " << std::sqrt(dot(error, error)) + std::sqrt(dot(error_face, error_face))
                << "\n";

      NodeValue<TinyVector<3>> ur3d{mesh->connectivity()};
      ur3d.fill(zero);

      parallel_for(
        mesh->numberOfNodes(), PUGS_LAMBDA(NodeId node_id) {
          TinyVector<Dimension> x = zero;
          const auto node_cells   = node_to_cell_matrix[node_id];
          for (size_t i_cell = 0; i_cell < node_cells.size(); ++i_cell) {
            CellId cell_id = node_cells[i_cell];
            x += w_rj(node_id, i_cell) * Uj[cell_id];
          }
          const auto node_faces = node_to_face_matrix[node_id];
          for (size_t i_face = 0; i_face < node_faces.size(); ++i_face) {
            FaceId face_id = node_faces[i_face];
            if (primal_face_is_on_boundary[face_id]) {
              x += w_rl(node_id, i_face) * Ul[face_id];
            }
          }
          for (size_t i = 0; i < Dimension; ++i) {
            ur3d[node_id][i] = x[i];
          }
        });
    }
  } else {
    throw NotImplementedError("not done in 1d");
  }
}

template <size_t Dimension>
class UnsteadyElasticity<Dimension>::DirichletBoundaryCondition
{
 private:
  const Array<const TinyVector<Dimension>> m_value_list;
  const Array<const FaceId> m_face_list;

 public:
  const Array<const FaceId>&
  faceList() const
  {
    return m_face_list;
  }

  const Array<const TinyVector<Dimension>>&
  valueList() const
  {
    return m_value_list;
  }

  DirichletBoundaryCondition(const Array<const FaceId>& face_list, const Array<const TinyVector<Dimension>>& value_list)
    : m_value_list{value_list}, m_face_list{face_list}
  {
    Assert(m_value_list.size() == m_face_list.size());
  }

  ~DirichletBoundaryCondition() = default;
};

template <size_t Dimension>
class UnsteadyElasticity<Dimension>::NormalStrainBoundaryCondition
{
 private:
  const Array<const TinyVector<Dimension>> m_value_list;
  const Array<const FaceId> m_face_list;

 public:
  const Array<const FaceId>&
  faceList() const
  {
    return m_face_list;
  }

  const Array<const TinyVector<Dimension>>&
  valueList() const
  {
    return m_value_list;
  }

  NormalStrainBoundaryCondition(const Array<const FaceId>& face_list,
                                const Array<const TinyVector<Dimension>>& value_list)
    : m_value_list{value_list}, m_face_list{face_list}
  {
    Assert(m_value_list.size() == m_face_list.size());
  }

  ~NormalStrainBoundaryCondition() = default;
};

template <size_t Dimension>
class UnsteadyElasticity<Dimension>::SymmetryBoundaryCondition
{
 private:
  const Array<const TinyVector<Dimension>> m_value_list;
  const Array<const FaceId> m_face_list;

 public:
  const Array<const FaceId>&
  faceList() const
  {
    return m_face_list;
  }

 public:
  SymmetryBoundaryCondition(const Array<const FaceId>& face_list) : m_face_list{face_list} {}

  ~SymmetryBoundaryCondition() = default;
};

template <size_t Dimension>
class UnsteadyElasticity<Dimension>::InterpolationWeightsManager
{
 private:
  std::shared_ptr<const Mesh<Connectivity<Dimension>>> m_mesh;
  FaceValue<bool> m_primal_face_is_on_boundary;
  NodeValue<bool> m_primal_node_is_on_boundary;
  FaceValue<bool> m_primal_face_is_symmetry;
  CellValuePerNode<double> m_w_rj;
  FaceValuePerNode<double> m_w_rl;

 public:
  InterpolationWeightsManager(std::shared_ptr<const Mesh<Connectivity<Dimension>>> mesh,
                              FaceValue<bool> primal_face_is_on_boundary,
                              NodeValue<bool> primal_node_is_on_boundary,
                              FaceValue<bool> primal_face_is_symmetry)
    : m_mesh(mesh),
      m_primal_face_is_on_boundary(primal_face_is_on_boundary),
      m_primal_node_is_on_boundary(primal_node_is_on_boundary),
      m_primal_face_is_symmetry(primal_face_is_symmetry)
  {}
  ~InterpolationWeightsManager() = default;
  CellValuePerNode<double>&
  wrj()
  {
    return m_w_rj;
  }
  FaceValuePerNode<double>&
  wrl()
  {
    return m_w_rl;
  }
  void
  compute()
  {
    using MeshDataType      = MeshData<Dimension>;
    MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(*m_mesh);

    const NodeValue<const TinyVector<Dimension>>& xr = m_mesh->xr();

    const FaceValue<const TinyVector<Dimension>>& xl = mesh_data.xl();
    const CellValue<const TinyVector<Dimension>>& xj = mesh_data.xj();
    const auto& node_to_cell_matrix                  = m_mesh->connectivity().nodeToCellMatrix();
    const auto& node_to_face_matrix                  = m_mesh->connectivity().nodeToFaceMatrix();
    const auto& face_to_cell_matrix                  = m_mesh->connectivity().faceToCellMatrix();

    CellValuePerNode<double> w_rj{m_mesh->connectivity()};
    FaceValuePerNode<double> w_rl{m_mesh->connectivity()};

    const NodeValuePerFace<const TinyVector<Dimension>> primal_nlr = mesh_data.nlr();
    auto project_to_face = [&](const TinyVector<Dimension>& x, const FaceId face_id) -> const TinyVector<Dimension> {
      TinyVector<Dimension> proj;
      const TinyVector<Dimension> nil = primal_nlr(face_id, 0);
      proj                            = x - dot((x - xl[face_id]), nil) * nil;
      return proj;
    };

    for (size_t i = 0; i < w_rl.numberOfValues(); ++i) {
      w_rl[i] = std::numeric_limits<double>::signaling_NaN();
    }

    for (NodeId i_node = 0; i_node < m_mesh->numberOfNodes(); ++i_node) {
      SmallVector<double> b{Dimension + 1};
      b[0] = 1;
      for (size_t i = 1; i < Dimension + 1; i++) {
        b[i] = xr[i_node][i - 1];
      }
      const auto& node_to_cell = node_to_cell_matrix[i_node];

      if (not m_primal_node_is_on_boundary[i_node]) {
        SmallMatrix<double> A{Dimension + 1, node_to_cell.size()};
        for (size_t j = 0; j < node_to_cell.size(); j++) {
          A(0, j) = 1;
        }
        for (size_t i = 1; i < Dimension + 1; i++) {
          for (size_t j = 0; j < node_to_cell.size(); j++) {
            const CellId J = node_to_cell[j];
            A(i, j)        = xj[J][i - 1];
          }
        }

        SmallVector<double> x{node_to_cell.size()};
        x = zero;

        LeastSquareSolver ls_solver;
        ls_solver.solveLocalSystem(A, x, b);

        for (size_t j = 0; j < node_to_cell.size(); j++) {
          w_rj(i_node, j) = x[j];
        }
      } else {
        int nb_face_used = 0;
        for (size_t i_face = 0; i_face < node_to_face_matrix[i_node].size(); ++i_face) {
          FaceId face_id = node_to_face_matrix[i_node][i_face];
          if (m_primal_face_is_on_boundary[face_id]) {
            nb_face_used++;
          }
        }
        SmallMatrix<double> A{Dimension + 1, node_to_cell.size() + nb_face_used};
        for (size_t j = 0; j < node_to_cell.size() + nb_face_used; j++) {
          A(0, j) = 1;
        }
        for (size_t i = 1; i < Dimension + 1; i++) {
          for (size_t j = 0; j < node_to_cell.size(); j++) {
            const CellId J = node_to_cell[j];
            A(i, j)        = xj[J][i - 1];
          }
        }
        for (size_t i = 1; i < Dimension + 1; i++) {
          int cpt_face = 0;
          for (size_t i_face = 0; i_face < node_to_face_matrix[i_node].size(); ++i_face) {
            FaceId face_id = node_to_face_matrix[i_node][i_face];
            if (m_primal_face_is_on_boundary[face_id]) {
              if (m_primal_face_is_symmetry[face_id]) {
                for (size_t j = 0; j < face_to_cell_matrix[face_id].size(); ++j) {
                  const CellId cell_id                 = face_to_cell_matrix[face_id][j];
                  TinyVector<Dimension> xproj          = project_to_face(xj[cell_id], face_id);
                  A(i, node_to_cell.size() + cpt_face) = xproj[i - 1];
                }
              } else {
                A(i, node_to_cell.size() + cpt_face) = xl[face_id][i - 1];
              }
              cpt_face++;
            }
          }
        }

        SmallVector<double> x{node_to_cell.size() + nb_face_used};
        x = zero;

        LeastSquareSolver ls_solver;
        ls_solver.solveLocalSystem(A, x, b);

        for (size_t j = 0; j < node_to_cell.size(); j++) {
          w_rj(i_node, j) = x[j];
        }
        int cpt_face = node_to_cell.size();
        for (size_t i_face = 0; i_face < node_to_face_matrix[i_node].size(); ++i_face) {
          FaceId face_id = node_to_face_matrix[i_node][i_face];
          if (m_primal_face_is_on_boundary[face_id]) {
            w_rl(i_node, i_face) = x[cpt_face++];
          }
        }
      }
    }
    m_w_rj = w_rj;
    m_w_rl = w_rl;
  }
};

template UnsteadyElasticity<1>::UnsteadyElasticity(
  std::shared_ptr<const IMesh>,
  const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const double&,
  const double&);

template UnsteadyElasticity<2>::UnsteadyElasticity(
  std::shared_ptr<const IMesh>,
  const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const double&,
  const double&);

template UnsteadyElasticity<3>::UnsteadyElasticity(
  std::shared_ptr<const IMesh>,
  const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const FunctionSymbolId&,
  const double&,
  const double&);
