#include <catch2/catch.hpp>

#include <Array.hpp>
#include <PugsAssert.hpp>
#include <Types.hpp>

#include <deque>
#include <list>
#include <set>
#include <unordered_set>
#include <valarray>
#include <vector>

// Instantiate to ensure full coverage is performed
template class Array<int>;

TEST_CASE("Array", "[utils]")
{
  Array<int> a(10);
  REQUIRE(a.size() == 10);

  for (size_t i = 0; i < a.size(); ++i) {
    a[i] = 2 * i;
  }

  REQUIRE(((a[0] == 0) and (a[1] == 2) and (a[2] == 4) and (a[3] == 6) and (a[4] == 8) and (a[5] == 10) and
           (a[6] == 12) and (a[7] == 14) and (a[8] == 16) and (a[9] == 18)));

  SECTION("checking for copies")
  {
    Array<const int> b{a};

    REQUIRE(((b[0] == 0) and (b[1] == 2) and (b[2] == 4) and (b[3] == 6) and (b[4] == 8) and (b[5] == 10) and
             (b[6] == 12) and (b[7] == 14) and (b[8] == 16) and (b[9] == 18)));

    Array<int> c{a};

    REQUIRE(((c[0] == 0) and (c[1] == 2) and (c[2] == 4) and (c[3] == 6) and (c[4] == 8) and (c[5] == 10) and
             (c[6] == 12) and (c[7] == 14) and (c[8] == 16) and (c[9] == 18)));

    Array<int> d = std::move(c);

    REQUIRE(((d[0] == 0) and (d[1] == 2) and (d[2] == 4) and (d[3] == 6) and (d[4] == 8) and (d[5] == 10) and
             (d[6] == 12) and (d[7] == 14) and (d[8] == 16) and (d[9] == 18)));
  }

  SECTION("checking for fill")
  {
    Array<int> b(10);
    b.fill(3);

    REQUIRE(((b[0] == 3) and (b[1] == 3) and (b[2] == 3) and (b[3] == 3) and (b[4] == 3) and (b[5] == 3) and
             (b[6] == 3) and (b[7] == 3) and (b[8] == 3) and (b[9] == 3)));
  }

  SECTION("checking for affectations (shallow copy)")
  {
    Array<const int> b;
    b = a;

    REQUIRE(((b[0] == 0) and (b[1] == 2) and (b[2] == 4) and (b[3] == 6) and (b[4] == 8) and (b[5] == 10) and
             (b[6] == 12) and (b[7] == 14) and (b[8] == 16) and (b[9] == 18)));

    Array<int> c;
    c = a;

    REQUIRE(((c[0] == 0) and (c[1] == 2) and (c[2] == 4) and (c[3] == 6) and (c[4] == 8) and (c[5] == 10) and
             (c[6] == 12) and (c[7] == 14) and (c[8] == 16) and (c[9] == 18)));

    Array<int> d;
    d = std::move(c);

    REQUIRE(((d[0] == 0) and (d[1] == 2) and (d[2] == 4) and (d[3] == 6) and (d[4] == 8) and (d[5] == 10) and
             (d[6] == 12) and (d[7] == 14) and (d[8] == 16) and (d[9] == 18)));
  }

  SECTION("checking for affectations (deep copy)")
  {
    Array<int> b(copy(a));

    REQUIRE(((b[0] == 0) and (b[1] == 2) and (b[2] == 4) and (b[3] == 6) and (b[4] == 8) and (b[5] == 10) and
             (b[6] == 12) and (b[7] == 14) and (b[8] == 16) and (b[9] == 18)));

    b.fill(2);

    REQUIRE(((a[0] == 0) and (a[1] == 2) and (a[2] == 4) and (a[3] == 6) and (a[4] == 8) and (a[5] == 10) and
             (a[6] == 12) and (a[7] == 14) and (a[8] == 16) and (a[9] == 18)));

    REQUIRE(((b[0] == 2) and (b[1] == 2) and (b[2] == 2) and (b[3] == 2) and (b[4] == 2) and (b[5] == 2) and
             (b[6] == 2) and (b[7] == 2) and (b[8] == 2) and (b[9] == 2)));

    Array<int> c;
    c = a;

    REQUIRE(((c[0] == 0) and (c[1] == 2) and (c[2] == 4) and (c[3] == 6) and (c[4] == 8) and (c[5] == 10) and
             (c[6] == 12) and (c[7] == 14) and (c[8] == 16) and (c[9] == 18)));

    c = copy(b);

    REQUIRE(((a[0] == 0) and (a[1] == 2) and (a[2] == 4) and (a[3] == 6) and (a[4] == 8) and (a[5] == 10) and
             (a[6] == 12) and (a[7] == 14) and (a[8] == 16) and (a[9] == 18)));

    REQUIRE(((c[0] == 2) and (c[1] == 2) and (c[2] == 2) and (c[3] == 2) and (c[4] == 2) and (c[5] == 2) and
             (c[6] == 2) and (c[7] == 2) and (c[8] == 2) and (c[9] == 2)));
  }

  SECTION("checking for std container conversion")
  {
    {
      std::vector<int> v{1, 2, 5, 3};
      {
        Array<int> v_array = convert_to_array(v);

        REQUIRE(v_array.size() == v.size());
        REQUIRE(((v_array[0] == 1) and (v_array[1] == 2) and (v_array[2] == 5) and (v_array[3] == 3)));
      }

      {
        Array<const int> v_array = convert_to_array(v);

        REQUIRE(v_array.size() == v.size());
        REQUIRE(((v_array[0] == 1) and (v_array[1] == 2) and (v_array[2] == 5) and (v_array[3] == 3)));
      }
    }

    {
      std::vector<int> w;
      {
        Array<int> w_array = convert_to_array(w);
        REQUIRE(w_array.size() == 0);
      }
      {
        Array<const int> w_array = convert_to_array(w);
        REQUIRE(w_array.size() == 0);
      }
    }

    {
      std::valarray<int> v{1, 2, 5, 3};
      Array<int> v_array = convert_to_array(v);

      REQUIRE(v_array.size() == v.size());
      REQUIRE(((v_array[0] == 1) and (v_array[1] == 2) and (v_array[2] == 5) and (v_array[3] == 3)));
    }

    {
      std::set<int> s{4, 2, 5, 3, 1, 3, 2};
      Array<int> s_array = convert_to_array(s);

      REQUIRE(s_array.size() == s.size());
      REQUIRE(
        ((s_array[0] == 1) and (s_array[1] == 2) and (s_array[2] == 3) and (s_array[3] == 4) and (s_array[4] == 5)));
    }

    {
      std::unordered_set<int> us{4, 2, 5, 3, 1, 3, 2};
      Array<int> us_array = convert_to_array(us);

      REQUIRE(us_array.size() == us.size());

      std::set<int> s;
      for (size_t i = 0; i < us_array.size(); ++i) {
        REQUIRE((us.find(us_array[i]) != us.end()));
        s.insert(us_array[i]);
      }
      REQUIRE(s.size() == us_array.size());
    }

    {
      std::multiset<int> ms{4, 2, 5, 3, 1, 3, 2};
      Array<int> ms_array = convert_to_array(ms);

      REQUIRE(ms_array.size() == ms.size());
      REQUIRE(((ms_array[0] == 1) and (ms_array[1] == 2) and (ms_array[2] == 2) and (ms_array[3] == 3) and
               (ms_array[4] == 3) and (ms_array[5] == 4) and (ms_array[6] == 5)));
    }

    {
      std::list<int> l{1, 3, 5, 6, 2};
      Array<int> l_array = convert_to_array(l);

      REQUIRE(l_array.size() == l.size());
      REQUIRE(
        ((l_array[0] == 1) and (l_array[1] == 3) and (l_array[2] == 5) and (l_array[3] == 6) and (l_array[4] == 2)));
    }

    {
      std::deque<int> q{1, 3, 5, 6, 2};
      q.push_front(2);
      Array<int> q_array = convert_to_array(q);

      REQUIRE(q_array.size() == q.size());
      REQUIRE(((q_array[0] == 2) and (q_array[1] == 1) and (q_array[2] == 3) and (q_array[3] == 5) and
               (q_array[4] == 6) and (q_array[5] == 2)));
    }
  }

#ifndef NDEBUG
  SECTION("checking for bounds violation")
  {
    REQUIRE_THROWS_AS(a[10], AssertError);
  }
#endif   // NDEBUG
}
