#include <catch2/catch.hpp>

#include <ASTNodeValueBuilder.hpp>

#include <ASTBuilder.hpp>
#include <ASTNodeDataTypeBuilder.hpp>

#include <ASTNodeDeclarationToAffectationConverter.hpp>
#include <ASTNodeTypeCleaner.hpp>

#include <ASTNodeExpressionBuilder.hpp>

#include <ASTNodeAffectationExpressionBuilder.hpp>

#include <ASTSymbolTableBuilder.hpp>

#include <ASTPrinter.hpp>

#include <Demangle.hpp>

#include <PEGGrammar.hpp>

#include <sstream>

#define CHECK_AFFECTATION_RESULT(data, variable_name, expected_value)         \
  {                                                                           \
    string_input input{data, "test.pgs"};                                     \
    auto ast = ASTBuilder::build(input);                                      \
                                                                              \
    ASTSymbolTableBuilder{*ast};                                              \
    ASTNodeDataTypeBuilder{*ast};                                             \
    ASTNodeValueBuilder{*ast};                                                \
                                                                              \
    ASTNodeDeclarationToAffectationConverter{*ast};                           \
    ASTNodeTypeCleaner<language::declaration>{*ast};                          \
                                                                              \
    ASTNodeExpressionBuilder{*ast};                                           \
    ExecUntilBreakOrContinue exec_policy;                                     \
    ast->execute(exec_policy);                                                \
                                                                              \
    auto symbol_table = ast->m_symbol_table;                                  \
                                                                              \
    using namespace TAO_PEGTL_NAMESPACE;                                      \
    position use_position{internal::iterator{"fixture"}, "fixture"};          \
    use_position.byte    = 10000;                                             \
    auto [symbol, found] = symbol_table->find(variable_name, use_position);   \
                                                                              \
    auto attributes = symbol->attributes();                                   \
    auto value      = std::get<decltype(expected_value)>(attributes.value()); \
                                                                              \
    REQUIRE(value == expected_value);                                         \
  }

#define CHECK_AFFECTATION_THROWS(data)                                                       \
  {                                                                                          \
    string_input input{data, "test.pgs"};                                                    \
    auto ast = ASTBuilder::build(input);                                                     \
                                                                                             \
    ASTSymbolTableBuilder{*ast};                                                             \
    ASTNodeDataTypeBuilder{*ast};                                                            \
    ASTNodeValueBuilder{*ast};                                                               \
                                                                                             \
    ASTNodeDeclarationToAffectationConverter{*ast};                                          \
    ASTNodeTypeCleaner<language::declaration>{*ast};                                         \
                                                                                             \
    REQUIRE_THROWS(ASTNodeExpressionBuilder{*ast},                                           \
                   Catch::Matchers::Contains("invalid operands to affectation expression")); \
  }

TEST_CASE("ASTAffectationProcessor", "[language]")
{
  SECTION("Affectations")
  {
    SECTION("B")
    {
      CHECK_AFFECTATION_RESULT("B b; b = true;", "b", true);
      CHECK_AFFECTATION_RESULT("N n = 1; B b; b = n;", "b", true);
      CHECK_AFFECTATION_RESULT("B b; b = 1;", "b", true);
      CHECK_AFFECTATION_RESULT("B b; b = 2.3;", "b", true);
    }

    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("N n = 1;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("N m = 2; N n = m;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("N n = true;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("N n = false;", "n", 0ul);
      CHECK_AFFECTATION_RESULT("N n = 2.3;", "n", 2ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("Z z = -1;", "z", -1l);
      CHECK_AFFECTATION_RESULT("Z z = true;", "z", 1l);
      CHECK_AFFECTATION_RESULT("Z z = false;", "z", 0l);
      CHECK_AFFECTATION_RESULT("Z z = -2.3;", "z", -2l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("R r = -1;", "r", -1.);
      CHECK_AFFECTATION_RESULT("R r = true;", "r", 1.);
      CHECK_AFFECTATION_RESULT("R r = false;", "r", 0.);
      CHECK_AFFECTATION_RESULT("R r = -2.3;", "r", -2.3);
    }
  }

  SECTION("+=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("N n = 1; n += 3;", "n", 4ul);
      CHECK_AFFECTATION_RESULT("N m = 2; N n = 1; n += m;", "n", 3ul);
      CHECK_AFFECTATION_RESULT("N n = 1; n += true;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("N n = 3; n += false;", "n", 3ul);
      CHECK_AFFECTATION_RESULT("N n = 2; n += 1.1;", "n", 3ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("Z z = 1; z += 3;", "z", 4l);
      CHECK_AFFECTATION_RESULT("N m = 2; Z z = 1; z += m;", "z", 3l);
      CHECK_AFFECTATION_RESULT("Z z = 1; z += true;", "z", 2l);
      CHECK_AFFECTATION_RESULT("Z z = 3; z += false;", "z", 3l);
      CHECK_AFFECTATION_RESULT("Z z = 2; z += 1.1;", "z", 3l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("R r = 1.2; r += 2.3;", "r", 3.5);
      CHECK_AFFECTATION_RESULT("N m = 2; R r = 1.3; r += m;", "r", 3.3);
      CHECK_AFFECTATION_RESULT("R r = 1.1; r += true;", "r", 2.1);
      CHECK_AFFECTATION_RESULT("R r = 3.3; r += false;", "r", 3.3);
      CHECK_AFFECTATION_RESULT("R r = 2; r += 1.1;", "r", 3.1);
    }
  }

  SECTION("-=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("N n = 3; n -= 2;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("N m = 2; N n = 4; n -= m;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("N n = 1; n -= true;", "n", 0ul);
      CHECK_AFFECTATION_RESULT("N n = 3; n -= false;", "n", 3ul);
      CHECK_AFFECTATION_RESULT("N n = 2; n -= 1.1;", "n", 0ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("Z z = 1; z -= 3;", "z", -2l);
      CHECK_AFFECTATION_RESULT("N m = 2; Z z = 1; z -= m;", "z", -1l);
      CHECK_AFFECTATION_RESULT("Z z = 1; z -= true;", "z", 0l);
      CHECK_AFFECTATION_RESULT("Z z = 3; z -= false;", "z", 3l);
      CHECK_AFFECTATION_RESULT("Z z = 2; z -= 2.1;", "z", 0l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("R r = 1.1; r -= 2;", "r", (1.1 - 2l));
      CHECK_AFFECTATION_RESULT("N m = 2; R r = 1.3; r -= m;", "r", (1.3 - 2ul));
      CHECK_AFFECTATION_RESULT("R r = 1.1; r -= true;", "r", (1.1 - true));
      CHECK_AFFECTATION_RESULT("R r = 3.3; r -= false;", "r", 3.3);
      CHECK_AFFECTATION_RESULT("R r = 2; r -= 1.1;", "r", (2. - 1.1));
    }
  }

  SECTION("*=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("N n = 3; n *= 2;", "n", 6ul);
      CHECK_AFFECTATION_RESULT("N m = 2; N n = 4; n *= m;", "n", 8ul);
      CHECK_AFFECTATION_RESULT("N n = 1; n *= true;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("N n = 3; n *= false;", "n", 0ul);
      CHECK_AFFECTATION_RESULT("N n = 2; n *= 2.51;", "n", 5ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("Z z = 1; z *= 3;", "z", 3l);
      CHECK_AFFECTATION_RESULT("N m = 2; Z z = -2; z *= m;", "z", -4l);
      CHECK_AFFECTATION_RESULT("Z z = 1; z *= true;", "z", 1l);
      CHECK_AFFECTATION_RESULT("Z z = 3; z *= false;", "z", 0l);
      CHECK_AFFECTATION_RESULT("Z z = 2; z *= -2.51;", "z", -5l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("R r = 1.1; r *= 2;", "r", (1.1 * 2l));
      CHECK_AFFECTATION_RESULT("N m = 2; R r = 1.3; r *= m;", "r", (1.3 * 2ul));
      CHECK_AFFECTATION_RESULT("R r = 1.1; r *= true;", "r", (1.1 * true));
      CHECK_AFFECTATION_RESULT("R r = 3.3; r *= false;", "r", (3.3 * false));
      CHECK_AFFECTATION_RESULT("R r = 2; r *= 1.1;", "r", (2. * 1.1));
    }
  }

  SECTION("/=")
  {
    SECTION("N")
    {
      CHECK_AFFECTATION_RESULT("N n = 4; n /= 2;", "n", 2ul);
      CHECK_AFFECTATION_RESULT("N m = 2; N n = 6; n /= m;", "n", 3ul);
      CHECK_AFFECTATION_RESULT("N n = 1; n /= true;", "n", 1ul);
      CHECK_AFFECTATION_RESULT("N n = 2; n /= 2.51;", "n", 0ul);
    }

    SECTION("Z")
    {
      CHECK_AFFECTATION_RESULT("Z z = 7; z /= -3;", "z", -2l);
      CHECK_AFFECTATION_RESULT("N m = 3; Z z = 6; z /= m;", "z", 2l);
      CHECK_AFFECTATION_RESULT("Z z = 6; z /= true;", "z", 6l);
      CHECK_AFFECTATION_RESULT("Z z = 4; z /= -2.;", "z", -2l);
    }

    SECTION("R")
    {
      CHECK_AFFECTATION_RESULT("R r = 1.1; r /= 2;", "r", (1.1 / 2l));
      CHECK_AFFECTATION_RESULT("N m = 2; R r = 1.3; r /= m;", "r", (1.3 / 2ul));
      CHECK_AFFECTATION_RESULT("R r = 1.1; r /= true;", "r", (1.1 / true));
      CHECK_AFFECTATION_RESULT("R r = 2; r /= 1.1;", "r", (2. / 1.1));
    }
  }

  SECTION("errors")
  {
    CHECK_AFFECTATION_THROWS("B b = true; b += 1;");
    CHECK_AFFECTATION_THROWS("B b = true; b *= 1;");
    CHECK_AFFECTATION_THROWS("B b = true; b -= 1;");
    CHECK_AFFECTATION_THROWS("B b = true; b /= 1;");
  }
}
