#include <catch2/catch.hpp>

#include <ASTBuilder.hpp>
#include <ASTSymbolTableBuilder.hpp>

#include <ASTSymbolInitializationChecker.hpp>

TEST_CASE("ASTSymbolInitializationChecker", "[language]")
{
  SECTION("Declarative initialization")
  {
    std::string_view data = R"(
N m = 2;
N n = m ;
N p;
)";

    string_input input{data, "test.pgs"};
    auto ast = ASTBuilder::build(input);

    ASTSymbolTableBuilder{*ast};
    ASTSymbolInitializationChecker{*ast};

    position position{internal::iterator{"fixture"}, "fixture"};
    position.byte = data.size();   // ensure that variables are declared at this point

    auto [symbol_m, found_m] = ast->m_symbol_table->find("m", position);
    REQUIRE(found_m);
    REQUIRE(symbol_m->attributes().isInitialized());

    auto [symbol_n, found_n] = ast->m_symbol_table->find("n", position);
    REQUIRE(found_n);
    REQUIRE(symbol_n->attributes().isInitialized());

    auto [symbol_p, found_p] = ast->m_symbol_table->find("p", position);
    REQUIRE(found_p);
    REQUIRE(not symbol_p->attributes().isInitialized());
  }

  SECTION("Declaration plus affectation")
  {
    std::string_view data = R"(
Z z;
N m;
N n;
n = 2;
m = n;
)";

    string_input input{data, "test.pgs"};
    auto ast = ASTBuilder::build(input);

    ASTSymbolTableBuilder{*ast};
    ASTSymbolInitializationChecker{*ast};

    position position{internal::iterator{"fixture"}, "fixture"};
    position.byte = data.size();   // ensure that variables are declared at this point

    auto [symbol_m, found_m] = ast->m_symbol_table->find("m", position);
    REQUIRE(found_m);
    REQUIRE(symbol_m->attributes().isInitialized());

    auto [symbol_n, found_n] = ast->m_symbol_table->find("n", position);
    REQUIRE(found_n);
    REQUIRE(symbol_n->attributes().isInitialized());

    auto [symbol_z, found_z] = ast->m_symbol_table->find("z", position);
    REQUIRE(found_z);
    REQUIRE(not symbol_z->attributes().isInitialized());
  }

  SECTION("Declarative initialization")
  {
    std::string_view data = R"(
let f: R->R, x->x+1;
)";

    string_input input{data, "test.pgs"};
    auto ast = ASTBuilder::build(input);

    ASTSymbolTableBuilder{*ast};
    ASTSymbolInitializationChecker{*ast};

    position position{internal::iterator{"fixture"}, "fixture"};
    position.byte = data.size();   // ensure that variables are declared at this point

    auto [symbol_m, found_m] = ast->m_symbol_table->find("f", position);
    REQUIRE(found_m);
    REQUIRE(symbol_m->attributes().isInitialized());
  }

  SECTION("errors")
  {
    SECTION("used uninitialized")
    {
      std::string_view data = R"(
N n;
N m = n;
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      REQUIRE_THROWS_AS(ASTSymbolInitializationChecker{*ast}, parse_error);
    }

    SECTION("used uninitialized in function")
    {
      std::string_view data = R"(
R y;
let f : R->R, x->x+y;
)";

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      REQUIRE_THROWS_AS(ASTSymbolInitializationChecker{*ast}, parse_error);
    }
  }
}
