#include <catch2/catch.hpp>

#include <ASTNodeValueBuilder.hpp>

#include <ASTBuilder.hpp>
#include <ASTNodeDataTypeBuilder.hpp>

#include <ASTSymbolTableBuilder.hpp>

#include <ASTPrinter.hpp>

#define CHECK_AST_VALUES(data, expected_output)                                                      \
  {                                                                                                  \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                   \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view>);        \
                                                                                                     \
    string_input input{data, "test.pgs"};                                                            \
    auto ast = ASTBuilder::build(input);                                                             \
                                                                                                     \
    ASTSymbolTableBuilder{*ast};                                                                     \
    ASTNodeDataTypeBuilder{*ast};                                                                    \
    ASTNodeValueBuilder{*ast};                                                                       \
                                                                                                     \
    std::stringstream ast_output;                                                                    \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::data_value}}; \
                                                                                                     \
    REQUIRE(ast_output.str() == expected_output);                                                    \
  }

TEST_CASE("ASTNodeValuebuilder", "[language]")
{
  SECTION("integer")
  {
    std::string_view data = R"(
01;
)";

    std::string_view result = R"(
(root:--)
 `-(language::integer:01:1)
)";

    CHECK_AST_VALUES(data, result);
  }

  SECTION("real")
  {
    std::string_view data = R"(
1.300;
)";

    std::string_view result = R"(
(root:--)
 `-(language::real:1.300:1.3)
)";

    CHECK_AST_VALUES(data, result);
  }

  SECTION("true")
  {
    std::string_view data = R"(
true;
)";

    std::string_view result = R"(
(root:--)
 `-(language::true_kw:1)
)";

    CHECK_AST_VALUES(data, result);
  }

  SECTION("false")
  {
    std::string_view data = R"(
false;
)";

    std::string_view result = R"(
(root:--)
 `-(language::false_kw:0)
)";

    CHECK_AST_VALUES(data, result);
  }

  SECTION("string")
  {
    std::string_view data = R"(
"\"foo\" or \"bar\"";
)";

    std::string_view result = R"(
(root:--)
 `-(language::literal:"\"foo\" or \"bar\"":"\"foo\" or \"bar\"")
)";

    CHECK_AST_VALUES(data, result);
  }

  SECTION("for empty test")
  {
    std::string_view data = R"(
for(;;);
)";

    std::string_view result = R"(
(root:--)
 `-(language::for_statement:--)
     +-(language::for_init:--)
     +-(language::for_test:1)
     +-(language::for_post:--)
     `-(language::for_statement_block:--)
)";

    CHECK_AST_VALUES(data, result);
  }

  SECTION("block values")
  {
    std::string_view data = R"(
{
  1;
  2.3;
}
)";

    std::string_view result = R"(
(root:--)
 `-(language::block:--)
     +-(language::integer:1:1)
     `-(language::real:2.3:2.3)
)";

    CHECK_AST_VALUES(data, result);
  }
}
