#include <catch2/catch.hpp>

#include <ASTNodeValueBuilder.hpp>

#include <ASTBuilder.hpp>
#include <ASTNodeDataTypeBuilder.hpp>

#include <ASTNodeTypeCleaner.hpp>

#include <ASTSymbolTableBuilder.hpp>

#include <ASTPrinter.hpp>

#define CHECK_AST(data, expected_output)                                                       \
  {                                                                                            \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);             \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view>);  \
                                                                                               \
    string_input input{data, "test.pgs"};                                                      \
    auto ast = ASTBuilder::build(input);                                                       \
                                                                                               \
    ASTSymbolTableBuilder{*ast};                                                               \
    ASTNodeDataTypeBuilder{*ast};                                                              \
    ASTNodeValueBuilder{*ast};                                                                 \
                                                                                               \
    ASTNodeTypeCleaner<language::declaration>{*ast};                                           \
                                                                                               \
    std::stringstream ast_output;                                                              \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::none}}; \
                                                                                               \
    REQUIRE(ast_output.str() == expected_output);                                              \
  }

TEST_CASE("ASTNodeTypeCleaner", "[language]")
{
  SECTION("no declaration")
  {
    std::string_view data = R"(
cout << "two=" << 2 << "\n";
)";

    std::string_view result = R"(
(root)
 `-(language::cout_kw)
     +-(language::literal:"two=")
     +-(language::integer:2)
     `-(language::literal:"\n")
)";

    CHECK_AST(data, result);
  }

  SECTION("single declaration")
  {
    std::string_view data = R"(
R z;
z = 1;
)";

    std::string_view result = R"(
(root)
 `-(language::eq_op)
     +-(language::name:z)
     `-(language::integer:1)
)";

    CHECK_AST(data, result);
  }

  SECTION("multiple declaration")
  {
    std::string_view data = R"(
Z z;
z = 1;
if (true) {
  R x;
  x = 0.5 *z;
}
)";

    std::string_view result = R"(
(root)
 +-(language::eq_op)
 |   +-(language::name:z)
 |   `-(language::integer:1)
 `-(language::if_statement)
     +-(language::true_kw)
     `-(language::block)
         `-(language::eq_op)
             +-(language::name:x)
             `-(language::multiply_op)
                 +-(language::real:0.5)
                 `-(language::name:z)
)";

    CHECK_AST(data, result);
  }
}
