#include <catch2/catch.hpp>

#include <ASTNodeValueBuilder.hpp>

#include <ASTBuilder.hpp>
#include <ASTNodeDataTypeBuilder.hpp>

#include <ASTNodeDeclarationToAffectationConverter.hpp>
#include <ASTNodeTypeCleaner.hpp>

#include <ASTNodeEmptyBlockCleaner.hpp>

#include <ASTNodeExpressionBuilder.hpp>

#include <ASTSymbolTableBuilder.hpp>

#include <ASTPrinter.hpp>

#include <PEGGrammar.hpp>

#include <Demangle.hpp>

#define CHECK_AST(data, expected_output)                                                            \
  {                                                                                                 \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                  \
    static_assert((std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view>) or    \
                  (std::is_same_v<std::decay_t<decltype(expected_output)>, std::string>));          \
                                                                                                    \
    string_input input{data, "test.pgs"};                                                           \
    auto ast = ASTBuilder::build(input);                                                            \
                                                                                                    \
    ASTSymbolTableBuilder{*ast};                                                                    \
    ASTNodeDataTypeBuilder{*ast};                                                                   \
    ASTNodeValueBuilder{*ast};                                                                      \
                                                                                                    \
    ASTNodeDeclarationToAffectationConverter{*ast};                                                 \
    ASTNodeTypeCleaner<language::declaration>{*ast};                                                \
                                                                                                    \
    ASTNodeEmptyBlockCleaner{*ast};                                                                 \
                                                                                                    \
    ASTNodeExpressionBuilder{*ast};                                                                 \
    std::stringstream ast_output;                                                                   \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::exec_type}}; \
                                                                                                    \
    REQUIRE(ast_output.str() == expected_output);                                                   \
  }

TEST_CASE("ASTNodeEmptyBlockCleaner", "[language]")
{
  SECTION("empty file")
  {
    std::string_view data = R"(
)";

    std::string_view result = R"(
(root:ASTNodeListProcessor)
)";

    CHECK_AST(data, result);
  }

  SECTION("keep non-empty block")
  {
    std::string_view data = R"(
{
  R x = 3;
}
)";

    std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::block:ASTNodeListProcessor)
     `-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
         +-(language::name:x:NameProcessor)
         `-(language::integer:3:FakeProcessor)
)";

    CHECK_AST(data, result);
  }

  SECTION("remove empty block")
  {
    std::string_view data = R"(
{
  R x;
}
)";

    std::string_view result = R"(
(root:ASTNodeListProcessor)
)";

    CHECK_AST(data, result);
  }

  SECTION("remove nested empty blocks")
  {
    std::string_view data = R"(
{
  R x;
  {
    R y;
  }
  {
    R z;
    {
      R w;
    }
  }
}
)";

    std::string_view result = R"(
(root:ASTNodeListProcessor)
)";

    CHECK_AST(data, result);
  }

  SECTION("remove nested empty blocks")
  {
    std::string_view data = R"(
{
  R x;
  {
    R y;
  }
  {
    R z;
    {
      R w = 4;
    }
  }
}
)";

    std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::block:ASTNodeListProcessor)
     `-(language::block:ASTNodeListProcessor)
         `-(language::block:ASTNodeListProcessor)
             `-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
                 +-(language::name:w:NameProcessor)
                 `-(language::integer:4:FakeProcessor)
)";

    CHECK_AST(data, result);
  }
}
