#include <catch2/catch.hpp>

#include <ASTNodeValueBuilder.hpp>

#include <ASTBuilder.hpp>
#include <ASTNodeDataTypeBuilder.hpp>

#include <ASTNodeDeclarationToAffectationConverter.hpp>

#include <ASTSymbolTableBuilder.hpp>

#include <ASTPrinter.hpp>

#define CHECK_AST(data, expected_output)                                                       \
  {                                                                                            \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);             \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view>);  \
                                                                                               \
    string_input input{data, "test.pgs"};                                                      \
    auto ast = ASTBuilder::build(input);                                                       \
                                                                                               \
    ASTSymbolTableBuilder{*ast};                                                               \
    ASTNodeDataTypeBuilder{*ast};                                                              \
    ASTNodeValueBuilder{*ast};                                                                 \
                                                                                               \
    ASTNodeDeclarationToAffectationConverter{*ast};                                            \
                                                                                               \
    std::stringstream ast_output;                                                              \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::none}}; \
                                                                                               \
    REQUIRE(ast_output.str() == expected_output);                                              \
  }

TEST_CASE("ASTNodeDeclarationToAffectationConverter", "[language]")
{
  SECTION("nothing to convert")
  {
    std::string_view data = R"(
R z;
)";

    std::string_view result = R"(
(root)
 `-(language::declaration)
     +-(language::R_set)
     `-(language::name:z)
)";

    CHECK_AST(data, result);
  }

  SECTION("simple constructor")
  {
    std::string_view data = R"(
R z = 0;
)";

    std::string_view result = R"(
(root)
 `-(language::eq_op)
     +-(language::name:z)
     `-(language::integer:0)
)";

    CHECK_AST(data, result);
  }

  SECTION("complex constructors")
  {
    std::string_view data = R"(
N k = 0;
for(N i=0; i<2; ++i) {
  N j = 2*i+k;
  k = 2*j-1;
}
)";

    std::string_view result = R"(
(root)
 +-(language::eq_op)
 |   +-(language::name:k)
 |   `-(language::integer:0)
 `-(language::for_statement)
     +-(language::eq_op)
     |   +-(language::name:i)
     |   `-(language::integer:0)
     +-(language::lesser_op)
     |   +-(language::name:i)
     |   `-(language::integer:2)
     +-(language::unary_plusplus)
     |   `-(language::name:i)
     `-(language::for_statement_block)
         +-(language::eq_op)
         |   +-(language::name:j)
         |   `-(language::plus_op)
         |       +-(language::multiply_op)
         |       |   +-(language::integer:2)
         |       |   `-(language::name:i)
         |       `-(language::name:k)
         `-(language::eq_op)
             +-(language::name:k)
             `-(language::minus_op)
                 +-(language::multiply_op)
                 |   +-(language::integer:2)
                 |   `-(language::name:j)
                 `-(language::integer:1)
)";

    CHECK_AST(data, result);
  }
}
