#include <catch2/catch.hpp>

#include <ASTNodeValueBuilder.hpp>

#include <ASTBuilder.hpp>
#include <ASTNodeDataTypeBuilder.hpp>

#include <ASTNodeDeclarationToAffectationConverter.hpp>
#include <ASTNodeTypeCleaner.hpp>

#include <ASTNodeExpressionBuilder.hpp>

#include <ASTNodeBinaryOperatorExpressionBuilder.hpp>

#include <ASTSymbolTableBuilder.hpp>

#include <ASTPrinter.hpp>

#include <Demangle.hpp>

#include <PEGGrammar.hpp>

#define CHECK_AST(data, expected_output)                                                            \
  {                                                                                                 \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                  \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view> or      \
                  std::is_same_v<std::decay_t<decltype(expected_output)>, std::string>);            \
                                                                                                    \
    string_input input{data, "test.pgs"};                                                           \
    auto ast = ASTBuilder::build(input);                                                            \
                                                                                                    \
    ASTSymbolTableBuilder{*ast};                                                                    \
    ASTNodeDataTypeBuilder{*ast};                                                                   \
    ASTNodeValueBuilder{*ast};                                                                      \
                                                                                                    \
    ASTNodeDeclarationToAffectationConverter{*ast};                                                 \
    ASTNodeTypeCleaner<language::declaration>{*ast};                                                \
                                                                                                    \
    ASTNodeExpressionBuilder{*ast};                                                                 \
                                                                                                    \
    std::stringstream ast_output;                                                                   \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::exec_type}}; \
                                                                                                    \
    REQUIRE(ast_output.str() == expected_output);                                                   \
  }

#define REQUIRE_AST_THROWS_WITH(data, expected_output)                                          \
  {                                                                                             \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);              \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string>);        \
                                                                                                \
    string_input input{data, "test.pgs"};                                                       \
    auto ast = ASTBuilder::build(input);                                                        \
                                                                                                \
    ASTSymbolTableBuilder{*ast};                                                                \
    ASTNodeDataTypeBuilder{*ast};                                                               \
    ASTNodeValueBuilder{*ast};                                                                  \
                                                                                                \
    ASTNodeDeclarationToAffectationConverter{*ast};                                             \
    ASTNodeTypeCleaner<language::declaration>{*ast};                                            \
                                                                                                \
    REQUIRE_THROWS(ASTNodeExpressionBuilder{*ast}, Catch::Matchers::Contains(expected_output)); \
  }

TEST_CASE("ASTNodeBinaryOperatorExpressionBuilder", "[language]")
{
  SECTION("multiply")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b*true;
false*b*true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:FakeProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, bool>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, bool, bool>)
     |   +-(language::false_kw:FakeProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n*m*n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, unsigned long, unsigned long>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a*3*a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, long>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:FakeProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3*1.2*2*false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, double, bool>)
     +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, double, long>)
     |   +-(language::multiply_op:BinaryExpressionProcessor<language::multiply_op, double, double>)
     |   |   +-(language::real:2.3:FakeProcessor)
     |   |   `-(language::real:1.2:FakeProcessor)
     |   `-(language::integer:2:FakeProcessor)
     `-(language::false_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("divide")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b/true;
false/b/true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:FakeProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, long, bool>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, bool, bool>)
     |   +-(language::false_kw:FakeProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n/m/n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, unsigned long, unsigned long>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a/3/a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, long, long>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:FakeProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3/1.2/2/false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::divide_op:BinaryExpressionProcessor<language::divide_op, double, bool>)
     +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, double, long>)
     |   +-(language::divide_op:BinaryExpressionProcessor<language::divide_op, double, double>)
     |   |   +-(language::real:2.3:FakeProcessor)
     |   |   `-(language::real:1.2:FakeProcessor)
     |   `-(language::integer:2:FakeProcessor)
     `-(language::false_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("plus")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b+true;
false+b+true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:FakeProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, long, bool>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, bool, bool>)
     |   +-(language::false_kw:FakeProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n+m+n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, unsigned long, unsigned long>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a+3+a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, long, long>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:FakeProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3+1.2+2+false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:BinaryExpressionProcessor<language::plus_op, double, bool>)
     +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, double, long>)
     |   +-(language::plus_op:BinaryExpressionProcessor<language::plus_op, double, double>)
     |   |   +-(language::real:2.3:FakeProcessor)
     |   |   `-(language::real:1.2:FakeProcessor)
     |   `-(language::integer:2:FakeProcessor)
     `-(language::false_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate bool")
    {
      std::string_view data = R"(
"foo"+true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<bool>)
     +-(language::literal:"foo":FakeProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate N")
    {
      std::string_view data = R"(
N n=0;
"foo"+n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, long>)
 |   +-(language::name:n:NameProcessor)
 |   `-(language::integer:0:FakeProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<unsigned long>)
     +-(language::literal:"foo":FakeProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate Z")
    {
      std::string_view data = R"(
"foo"+1;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<long>)
     +-(language::literal:"foo":FakeProcessor)
     `-(language::integer:1:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate R")
    {
      std::string_view data = R"(
"foo"+1.2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<double>)
     +-(language::literal:"foo":FakeProcessor)
     `-(language::real:1.2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string concatenate string")
    {
      std::string_view data = R"(
"foo"+"bar";
)";

      std::string string_name = demangle(typeid(std::string{}).name());

      std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::plus_op:ConcatExpressionProcessor<)" +
                           string_name + R"( >)
     +-(language::literal:"foo":FakeProcessor)
     `-(language::literal:"bar":FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("minus")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b-true;
false-b-true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:FakeProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, long, bool>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, bool, bool>)
     |   +-(language::false_kw:FakeProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n-m-n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, unsigned long, unsigned long>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a-3-a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, long, long>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:FakeProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3-1.2-2-false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::minus_op:BinaryExpressionProcessor<language::minus_op, double, bool>)
     +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, double, long>)
     |   +-(language::minus_op:BinaryExpressionProcessor<language::minus_op, double, double>)
     |   |   +-(language::real:2.3:FakeProcessor)
     |   |   `-(language::real:1.2:FakeProcessor)
     |   `-(language::integer:2:FakeProcessor)
     `-(language::false_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("or")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b or true;
false or b or true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:FakeProcessor)
 `-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, bool>)
     +-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, bool>)
     |   +-(language::false_kw:FakeProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n or m or n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, unsigned long>)
     +-(language::or_op:BinaryExpressionProcessor<language::or_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a or 3 or a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, long>)
     +-(language::or_op:BinaryExpressionProcessor<language::or_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:FakeProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3 or 1.2 or 2 or false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, bool>)
     +-(language::or_op:BinaryExpressionProcessor<language::or_op, bool, long>)
     |   +-(language::or_op:BinaryExpressionProcessor<language::or_op, double, double>)
     |   |   +-(language::real:2.3:FakeProcessor)
     |   |   `-(language::real:1.2:FakeProcessor)
     |   `-(language::integer:2:FakeProcessor)
     `-(language::false_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("and")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b and true;
false and b and true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:FakeProcessor)
 `-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, bool>)
     +-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, bool>)
     |   +-(language::false_kw:FakeProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n and m and n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, unsigned long>)
     +-(language::and_op:BinaryExpressionProcessor<language::and_op, unsigned long, unsigned long>)
     |   +-(language::name:n:NameProcessor)
     |   `-(language::name:m:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a and 3 and a;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, long>)
     +-(language::and_op:BinaryExpressionProcessor<language::and_op, long, long>)
     |   +-(language::name:a:NameProcessor)
     |   `-(language::integer:3:FakeProcessor)
     `-(language::name:a:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3 and 1.2 and 2 and false;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, bool>)
     +-(language::and_op:BinaryExpressionProcessor<language::and_op, bool, long>)
     |   +-(language::and_op:BinaryExpressionProcessor<language::and_op, double, double>)
     |   |   +-(language::real:2.3:FakeProcessor)
     |   |   `-(language::real:1.2:FakeProcessor)
     |   `-(language::integer:2:FakeProcessor)
     `-(language::false_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("xor")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b xor true;
false xor b xor true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::xor_op:BinaryExpressionProcessor<language::xor_op, bool, bool>)
 |   +-(language::name:b:NameProcessor)
 |   `-(language::true_kw:FakeProcessor)
 `-(language::xor_op:BinaryExpressionProcessor<language::xor_op, bool, bool>)
     +-(language::xor_op:BinaryExpressionProcessor<language::xor_op, bool, bool>)
     |   +-(language::false_kw:FakeProcessor)
     |   `-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n xor m;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::xor_op:BinaryExpressionProcessor<language::xor_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

      CHECK_AST(data, result);
      // std::string error = "invalid operands to binary expression";
      // REQUIRE_AST_THROWS_WITH(data, error);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a xor 3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::xor_op:BinaryExpressionProcessor<language::xor_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3 xor 1.2;
)";

      std::string error = "invalid operands to binary expression";
      REQUIRE_AST_THROWS_WITH(data, error);
    }
  }

  SECTION("greater")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b > true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n > m;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a > 3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3 > 1.2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_op:BinaryExpressionProcessor<language::greater_op, double, double>)
     +-(language::real:2.3:FakeProcessor)
     `-(language::real:1.2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("lesser")
  {
    SECTION("B")
    {
      std::string_view data = R"(
B b;
b < true;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("N")
    {
      std::string_view data = R"(
N n;
N m;
n < m;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("Z")
    {
      std::string_view data = R"(
Z a;
a < 3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R")
    {
      std::string_view data = R"(
2.3 < 1.2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_op:BinaryExpressionProcessor<language::lesser_op, double, double>)
     +-(language::real:2.3:FakeProcessor)
     `-(language::real:1.2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("greater or equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
B b;
b >= true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
N n;
N m;
n >= m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
Z a;
a >= 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 >= 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::greater_or_eq_op:BinaryExpressionProcessor<language::greater_or_eq_op, double, double>)
     +-(language::real:2.3:FakeProcessor)
     `-(language::real:1.2:FakeProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("lesser or equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
B b;
b <= true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
N n;
N m;
n <= m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
Z a;
a <= 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 <= 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::lesser_or_eq_op:BinaryExpressionProcessor<language::lesser_or_eq_op, double, double>)
     +-(language::real:2.3:FakeProcessor)
     `-(language::real:1.2:FakeProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("equal equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
B b;
b == true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
N n;
N m;
n == m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
Z a;
a == 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 == 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, double, double>)
     +-(language::real:2.3:FakeProcessor)
     `-(language::real:1.2:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string == string")
      {
        std::string_view data = R"(
"foo" == "bar";
)";

        std::string string_name = demangle(typeid(std::string{}).name());

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eqeq_op:BinaryExpressionProcessor<language::eqeq_op, )" +
                             string_name + ", " + string_name + R"( >)
     +-(language::literal:"foo":FakeProcessor)
     `-(language::literal:"bar":FakeProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("not equal")
    {
      SECTION("B")
      {
        std::string_view data = R"(
B b;
b != true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N")
      {
        std::string_view data = R"(
N n;
N m;
n != m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z")
      {
        std::string_view data = R"(
Z a;
a != 3;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, long, long>)
     +-(language::name:a:NameProcessor)
     `-(language::integer:3:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R")
      {
        std::string_view data = R"(
2.3 != 1.2;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, double, double>)
     +-(language::real:2.3:FakeProcessor)
     `-(language::real:1.2:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string != string")
      {
        std::string_view data = R"(
"foo" != "bar";
)";

        std::string string_name = demangle(typeid(std::string{}).name());

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::not_eq_op:BinaryExpressionProcessor<language::not_eq_op, )" +
                             string_name + ", " + string_name + R"( >)
     +-(language::literal:"foo":FakeProcessor)
     `-(language::literal:"bar":FakeProcessor)
)";

        CHECK_AST(data, result);
      }
    }
  }

  SECTION("Errors")
  {
    SECTION("Invalid binary operator type")
    {
      auto ast = std::make_unique<ASTNode>();

      REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "unexpected error: undefined binary operator");
    }

    SECTION("Invalid string binary operators")
    {
      SECTION("lhs bad multiply")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::multiply_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string multiply")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::multiply_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string multiply")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::multiply_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad divide")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::divide_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string divide")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::divide_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string divide")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::divide_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad plus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::plus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string plus bad rhs")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::plus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::void_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string plus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::plus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad minus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::minus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string minus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::minus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string minus")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::minus_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad or")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::or_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string or")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::or_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string or")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::or_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad and")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::and_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string and")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::and_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string and")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::and_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad xor")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::xor_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string xor")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::xor_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string xor")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::xor_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad >")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string >")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string >")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad <")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string <")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string <")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad >=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string >=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string >=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::greater_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad <=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string <=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string <=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::lesser_or_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::eqeq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::eqeq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::eqeq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("lhs bad !=")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::not_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::void_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("left string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::not_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::string_t;
        ast->children[1]->m_data_type = ASTNodeDataType::int_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }

      SECTION("right string ==")
      {
        auto ast = std::make_unique<ASTNode>();
        ast->set_type<language::not_eq_op>();
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children.emplace_back(std::make_unique<ASTNode>());
        ast->children[0]->m_data_type = ASTNodeDataType::int_t;
        ast->children[1]->m_data_type = ASTNodeDataType::string_t;

        REQUIRE_THROWS_WITH(ASTNodeBinaryOperatorExpressionBuilder{*ast}, "undefined operand type for binary operator");
      }
    }
  }
}
