#include <catch2/catch.hpp>

#include <ASTNodeValueBuilder.hpp>

#include <ASTBuilder.hpp>
#include <ASTNodeDataTypeBuilder.hpp>

#include <ASTNodeDeclarationToAffectationConverter.hpp>
#include <ASTNodeTypeCleaner.hpp>

#include <ASTNodeExpressionBuilder.hpp>

#include <ASTNodeAffectationExpressionBuilder.hpp>

#include <ASTSymbolTableBuilder.hpp>

#include <ASTPrinter.hpp>

#include <Demangle.hpp>

#include <PEGGrammar.hpp>

#define CHECK_AST(data, expected_output)                                                            \
  {                                                                                                 \
    static_assert(std::is_same_v<std::decay_t<decltype(data)>, std::string_view>);                  \
    static_assert(std::is_same_v<std::decay_t<decltype(expected_output)>, std::string_view> or      \
                  std::is_same_v<std::decay_t<decltype(expected_output)>, std::string>);            \
                                                                                                    \
    string_input input{data, "test.pgs"};                                                           \
    auto ast = ASTBuilder::build(input);                                                            \
                                                                                                    \
    ASTSymbolTableBuilder{*ast};                                                                    \
    ASTNodeDataTypeBuilder{*ast};                                                                   \
    ASTNodeValueBuilder{*ast};                                                                      \
                                                                                                    \
    ASTNodeDeclarationToAffectationConverter{*ast};                                                 \
    ASTNodeTypeCleaner<language::declaration>{*ast};                                                \
                                                                                                    \
    ASTNodeExpressionBuilder{*ast};                                                                 \
                                                                                                    \
    std::stringstream ast_output;                                                                   \
    ast_output << '\n' << ASTPrinter{*ast, ASTPrinter::Format::raw, {ASTPrinter::Info::exec_type}}; \
                                                                                                    \
    REQUIRE(ast_output.str() == expected_output);                                                   \
  }

TEST_CASE("ASTNodeAffectationExpressionBuilder", "[language]")
{
  SECTION("Affectations")
  {
    SECTION("boolean affectation")
    {
      SECTION("B <- B")
      {
        std::string_view data = R"(
B b=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, bool, bool>)
     +-(language::name:b:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("B <- N")
      {
        std::string_view data = R"(
N n; B b=n;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, bool, unsigned long>)
     +-(language::name:b:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("B <- Z")
      {
        std::string_view data = R"(
Z z; B b=z;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, bool, long>)
     +-(language::name:b:NameProcessor)
     `-(language::name:z:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("B <- R")
      {
        std::string_view data = R"(
R r; B b=r;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, bool, double>)
     +-(language::name:b:NameProcessor)
     `-(language::name:r:NameProcessor)
)";

        CHECK_AST(data, result);
      }
    }

    SECTION("unsigned integer affectation")
    {
      SECTION("N <- B")
      {
        std::string_view data = R"(
N n=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, bool>)
     +-(language::name:n:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N <- N")
      {
        std::string_view data = R"(
N m; N n=m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N <- Z")
      {
        std::string_view data = R"(
Z z; N n=z;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:z:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N <- R")
      {
        std::string_view data = R"(
R r; N n=r;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, double>)
     +-(language::name:n:NameProcessor)
     `-(language::name:r:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("N <- string (invalid)")
      {
        std::string_view data = R"(
N n="foo";
)";

        string_input input{data, "test.pgs"};
        auto ast = ASTBuilder::build(input);

        ASTSymbolTableBuilder{*ast};
        ASTNodeDataTypeBuilder{*ast};
        ASTNodeValueBuilder{*ast};

        ASTNodeDeclarationToAffectationConverter{*ast};
        ASTNodeTypeCleaner<language::declaration>{*ast};

        REQUIRE_THROWS_AS(ASTNodeExpressionBuilder{*ast}, parse_error);
      }
    }

    SECTION("integer affectation")
    {
      SECTION("Z <- B")
      {
        std::string_view data = R"(
Z z=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, long, bool>)
     +-(language::name:z:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z <- N")
      {
        std::string_view data = R"(
N m; Z z=m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, long, unsigned long>)
     +-(language::name:z:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z <- Z")
      {
        std::string_view data = R"(
Z q; Z z=q;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::name:q:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("B <- R")
      {
        std::string_view data = R"(
R r; Z z=r;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, long, double>)
     +-(language::name:z:NameProcessor)
     `-(language::name:r:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("Z <- string (invalid)")
      {
        std::string_view data = R"(
Z z="foo";
)";

        string_input input{data, "test.pgs"};
        auto ast = ASTBuilder::build(input);

        ASTSymbolTableBuilder{*ast};
        ASTNodeDataTypeBuilder{*ast};
        ASTNodeValueBuilder{*ast};

        ASTNodeDeclarationToAffectationConverter{*ast};
        ASTNodeTypeCleaner<language::declaration>{*ast};

        REQUIRE_THROWS_AS(ASTNodeExpressionBuilder{*ast}, parse_error);
      }
    }

    SECTION("double affectation")
    {
      SECTION("R <- B")
      {
        std::string_view data = R"(
R r=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, bool>)
     +-(language::name:r:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R <- N")
      {
        std::string_view data = R"(
N m; R r=m;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, unsigned long>)
     +-(language::name:r:NameProcessor)
     `-(language::name:m:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R <- Z")
      {
        std::string_view data = R"(
Z z; R r=z;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
     +-(language::name:r:NameProcessor)
     `-(language::name:z:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R <- R")
      {
        std::string_view data = R"(
R s; R r=s;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationProcessor<language::eq_op, double, double>)
     +-(language::name:r:NameProcessor)
     `-(language::name:s:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("R <- string (invalid)")
      {
        std::string_view data = R"(
R r="foo";
)";

        string_input input{data, "test.pgs"};
        auto ast = ASTBuilder::build(input);

        ASTSymbolTableBuilder{*ast};
        ASTNodeDataTypeBuilder{*ast};
        ASTNodeValueBuilder{*ast};

        ASTNodeDeclarationToAffectationConverter{*ast};
        ASTNodeTypeCleaner<language::declaration>{*ast};

        REQUIRE_THROWS_AS(ASTNodeExpressionBuilder{*ast}, parse_error);
      }
    }

    SECTION("string affectation")
    {
      SECTION("string <- B")
      {
        std::string_view data = R"(
string s=true;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationToStringProcessor<language::eq_op, bool>)
     +-(language::name:s:NameProcessor)
     `-(language::true_kw:FakeProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- N")
      {
        std::string_view data = R"(
N n; string s=n;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationToStringProcessor<language::eq_op, unsigned long>)
     +-(language::name:s:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- Z")
      {
        std::string_view data = R"(
Z z; string s=z;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationToStringProcessor<language::eq_op, long>)
     +-(language::name:s:NameProcessor)
     `-(language::name:z:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- R")
      {
        std::string_view data = R"(
R r; string s=r;
)";

        std::string_view result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationToStringProcessor<language::eq_op, double>)
     +-(language::name:s:NameProcessor)
     `-(language::name:r:NameProcessor)
)";

        CHECK_AST(data, result);
      }

      SECTION("string <- string")
      {
        std::string_view data = R"(
string s="foo";
)";

        std::string string_name = demangle(typeid(std::string{}).name());

        std::string result = R"(
(root:ASTNodeListProcessor)
 `-(language::eq_op:AffectationToStringProcessor<language::eq_op, )" +
                             string_name + R"( >)
     +-(language::name:s:NameProcessor)
     `-(language::literal:"foo":FakeProcessor)
)";

        CHECK_AST(data, result);
      }
    }
  }

  SECTION("+=")
  {
    SECTION("N += N")
    {
      std::string_view data = R"(
N n=1; n+=n;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, unsigned long, long>)
 |   +-(language::name:n:NameProcessor)
 |   `-(language::integer:1:FakeProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, unsigned long, unsigned long>)
     +-(language::name:n:NameProcessor)
     `-(language::name:n:NameProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R += N")
    {
      std::string_view data = R"(
R x=1; x+=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:FakeProcessor)
 `-(language::pluseq_op:AffectationProcessor<language::pluseq_op, double, long>)
     +-(language::name:x:NameProcessor)
     `-(language::integer:2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string += N")
    {
      std::string_view data = R"(
string s="foo"; s+=2;
)";

      std::string string_name = demangle(typeid(std::string{}).name());

      std::string result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationToStringProcessor<language::eq_op, )" +
                           string_name + R"( >)
 |   +-(language::name:s:NameProcessor)
 |   `-(language::literal:"foo":FakeProcessor)
 `-(language::pluseq_op:AffectationToStringProcessor<language::pluseq_op, long>)
     +-(language::name:s:NameProcessor)
     `-(language::integer:2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }
  }

  SECTION("-=")
  {
    SECTION("Z -= Z")
    {
      std::string_view data = R"(
Z z=1; z-=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
 |   +-(language::name:z:NameProcessor)
 |   `-(language::integer:1:FakeProcessor)
 `-(language::minuseq_op:AffectationProcessor<language::minuseq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::integer:2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R -= R")
    {
      std::string_view data = R"(
R x=1; x-=2.3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:FakeProcessor)
 `-(language::minuseq_op:AffectationProcessor<language::minuseq_op, double, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:2.3:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string -= string")
    {
      std::string_view data = R"(
string s="foo"; s-="bar";
)";

      std::string string_name = demangle(typeid(std::string{}).name());

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};
      ASTNodeValueBuilder{*ast};

      ASTNodeDeclarationToAffectationConverter{*ast};
      ASTNodeTypeCleaner<language::declaration>{*ast};

      REQUIRE_THROWS_AS(ASTNodeExpressionBuilder{*ast}, parse_error);
    }
  }

  SECTION("*=")
  {
    SECTION("Z *= Z")
    {
      std::string_view data = R"(
Z z=1; z*=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
 |   +-(language::name:z:NameProcessor)
 |   `-(language::integer:1:FakeProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::integer:2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R *= R")
    {
      std::string_view data = R"(
R x=1; x*=2.3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:FakeProcessor)
 `-(language::multiplyeq_op:AffectationProcessor<language::multiplyeq_op, double, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:2.3:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string *= Z")
    {
      std::string_view data = R"(
string s="foo"; s*=2;
)";

      std::string string_name = demangle(typeid(std::string{}).name());

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};
      ASTNodeValueBuilder{*ast};

      ASTNodeDeclarationToAffectationConverter{*ast};
      ASTNodeTypeCleaner<language::declaration>{*ast};

      REQUIRE_THROWS_AS(ASTNodeExpressionBuilder{*ast}, parse_error);
    }
  }

  SECTION("/=")
  {
    SECTION("Z /= Z")
    {
      std::string_view data = R"(
Z z=6; z/=2;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, long, long>)
 |   +-(language::name:z:NameProcessor)
 |   `-(language::integer:6:FakeProcessor)
 `-(language::divideeq_op:AffectationProcessor<language::divideeq_op, long, long>)
     +-(language::name:z:NameProcessor)
     `-(language::integer:2:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("R /= R")
    {
      std::string_view data = R"(
R x=1; x/=2.3;
)";

      std::string_view result = R"(
(root:ASTNodeListProcessor)
 +-(language::eq_op:AffectationProcessor<language::eq_op, double, long>)
 |   +-(language::name:x:NameProcessor)
 |   `-(language::integer:1:FakeProcessor)
 `-(language::divideeq_op:AffectationProcessor<language::divideeq_op, double, double>)
     +-(language::name:x:NameProcessor)
     `-(language::real:2.3:FakeProcessor)
)";

      CHECK_AST(data, result);
    }

    SECTION("string /= string")
    {
      std::string_view data = R"(
string s="foo"; s/="bar";
)";

      std::string string_name = demangle(typeid(std::string{}).name());

      string_input input{data, "test.pgs"};
      auto ast = ASTBuilder::build(input);

      ASTSymbolTableBuilder{*ast};
      ASTNodeDataTypeBuilder{*ast};
      ASTNodeValueBuilder{*ast};

      ASTNodeDeclarationToAffectationConverter{*ast};
      ASTNodeTypeCleaner<language::declaration>{*ast};

      REQUIRE_THROWS_AS(ASTNodeExpressionBuilder{*ast}, parse_error);
    }
  }

  SECTION("Errors")
  {
    SECTION("Invalid affectation operator")
    {
      auto ast = std::make_unique<ASTNode>();
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: undefined affectation operator");
    }

    SECTION("Invalid lhs")
    {
      auto ast = std::make_unique<ASTNode>();
      ast->set_type<language::eq_op>();
      ast->children.emplace_back(std::make_unique<ASTNode>());
      ast->children.emplace_back(std::make_unique<ASTNode>());
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: undefined value type for affectation");
    }

    SECTION("Invalid rhs")
    {
      auto ast = std::make_unique<ASTNode>();
      ast->set_type<language::eq_op>();
      ast->m_data_type = ASTNodeDataType::int_t;

      ast->children.emplace_back(std::make_unique<ASTNode>());
      ast->children.emplace_back(std::make_unique<ASTNode>());
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: undefined operand type for affectation");
    }

    SECTION("Invalid string rhs")
    {
      auto ast = std::make_unique<ASTNode>();
      ast->set_type<language::eq_op>();
      ast->m_data_type = ASTNodeDataType::string_t;

      ast->children.emplace_back(std::make_unique<ASTNode>());
      ast->children.emplace_back(std::make_unique<ASTNode>());
      REQUIRE_THROWS_WITH(ASTNodeAffectationExpressionBuilder{*ast},
                          "unexpected error: undefined operand type for string affectation");
    }
  }
}
