#ifndef FUNCTION_PROCESSOR_HPP
#define FUNCTION_PROCESSOR_HPP

#include <FunctionTable.hpp>
#include <SymbolTable.hpp>

#include <PEGGrammar.hpp>

#include <node_processor/INodeProcessor.hpp>

template <typename ProvidedValueType, typename ExpectedValueType>
class FunctionArgumentProcessor final : public INodeProcessor
{
 private:
  ASTNode& m_provided_value_node;
  ASTNodeDataVariant& m_symbol_value;

 public:
  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    m_provided_value_node.execute(exec_policy);

    if constexpr (std::is_same_v<ExpectedValueType, ProvidedValueType>) {
      m_symbol_value = m_provided_value_node.m_value;
    } else {
      m_symbol_value = static_cast<ExpectedValueType>(std::get<ProvidedValueType>(m_provided_value_node.m_value));
    }
  }

  FunctionArgumentProcessor(ASTNode& provided_value_node, SymbolTable::Symbol& argument_symbol)
    : m_provided_value_node{provided_value_node}, m_symbol_value{argument_symbol.attributes().value()}
  {}
};

template <typename ReturnType, typename ExpressionValueType>
class FunctionExpressionProcessor final : public INodeProcessor
{
 private:
  ASTNode& m_node;
  ASTNode& m_function_expression;

 public:
  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    m_function_expression.execute(exec_policy);

    if constexpr (std::is_same_v<ReturnType, ExpressionValueType>) {
      m_node.m_value = m_function_expression.m_value;
    } else {
      m_node.m_value = static_cast<ReturnType>(std::get<ExpressionValueType>(m_function_expression.m_value));
    }
  }

  FunctionExpressionProcessor(ASTNode& node)
    : m_node{node}, m_function_expression{[&]() -> ASTNode& {
        auto [i_symbol, found] = m_node.m_symbol_table->find(m_node.children[0]->string(), m_node.begin());
        Assert(found);
        uint64_t function_id = std::get<uint64_t>(i_symbol->attributes().value());

        FunctionDescriptor& function_descriptor = m_node.m_symbol_table->functionTable()[function_id];
        return *function_descriptor.definitionNode().children[1];
      }()}
  {}
};

class FunctionProcessor : public INodeProcessor
{
 private:
  std::vector<std::unique_ptr<INodeProcessor>> m_argument_processors;
  std::vector<std::unique_ptr<INodeProcessor>> m_function_expression_processors;

 public:
  void
  addArgumentProcessor(std::unique_ptr<INodeProcessor>&& argument_processor)
  {
    m_argument_processors.emplace_back(std::move(argument_processor));
  }

  void
  addFunctionExpressionProcessor(std::unique_ptr<INodeProcessor>&& function_processor)
  {
    m_function_expression_processors.emplace_back(std::move(function_processor));
  }

  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    for (auto& argument_processor : m_argument_processors) {
      argument_processor->execute(exec_policy);
    }

    for (auto& function_expression_processor : m_function_expression_processors) {
      function_expression_processor->execute(exec_policy);
    }
  }

  FunctionProcessor() = default;
};

#endif   // FUNCTION_PROCESSOR_HPP
