#ifndef CFUNCTION_PROCESSOR_HPP
#define CFUNCTION_PROCESSOR_HPP

#include <PEGGrammar.hpp>

#include <node_processor/INodeProcessor.hpp>

#include <CFunctionEmbedder.hpp>

template <typename ProvidedValueType, typename ExpectedValueType>
class CFunctionArgumentProcessor final : public INodeProcessor
{
 private:
  ASTNode& m_provided_value_node;
  ASTNodeDataVariant& m_argument_value;

 public:
  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    m_provided_value_node.execute(exec_policy);

    if constexpr (std::is_same_v<ExpectedValueType, ProvidedValueType>) {
      m_argument_value = m_provided_value_node.m_value;
    } else {
      m_argument_value = static_cast<ExpectedValueType>(std::get<ProvidedValueType>(m_provided_value_node.m_value));
    }
  }

  CFunctionArgumentProcessor(ASTNode& provided_value_node, ASTNodeDataVariant& argument_value)
    : m_provided_value_node{provided_value_node}, m_argument_value{argument_value}
  {}
};

class CFunctionExpressionProcessor final : public INodeProcessor
{
 private:
  ASTNode& m_node;

  std::shared_ptr<ICFunctionEmbedder> m_embedded_c_function;
  std::vector<ASTNodeDataVariant>& m_argument_values;

 public:
  void
  execute(ExecUntilBreakOrContinue&)
  {
    m_embedded_c_function->apply(m_argument_values, m_node.m_value);
  }

  CFunctionExpressionProcessor(ASTNode& node,
                               std::shared_ptr<ICFunctionEmbedder> embedded_c_function,
                               std::vector<ASTNodeDataVariant>& argument_values)
    : m_node{node}, m_embedded_c_function(embedded_c_function), m_argument_values{argument_values}
  {}
};

class CFunctionProcessor : public INodeProcessor
{
 private:
  std::unique_ptr<INodeProcessor> m_function_expression_processor;

  std::vector<std::unique_ptr<INodeProcessor>> m_argument_processors;
  std::vector<ASTNodeDataVariant> m_argument_values;

 public:
  void
  setNumberOfArguments(const size_t& number_of_arguments)
  {
    Assert(m_argument_values.size() == 0, "argument number has already been provided");
    m_argument_values.resize(number_of_arguments);
  }

  std::vector<ASTNodeDataVariant>&
  argumentValues()
  {
    return m_argument_values;
  }

  void
  addArgumentProcessor(std::unique_ptr<INodeProcessor>&& argument_processor)
  {
    m_argument_processors.emplace_back(std::move(argument_processor));
  }

  void
  setFunctionExpressionProcessor(std::unique_ptr<INodeProcessor>&& function_processor)
  {
    m_function_expression_processor = std::move(function_processor);
  }

  void
  execute(ExecUntilBreakOrContinue& exec_policy)
  {
    Assert(m_argument_processors.size() == m_argument_values.size());
    for (auto& argument_processor : m_argument_processors) {
      argument_processor->execute(exec_policy);
    }

    m_function_expression_processor->execute(exec_policy);
  }

  CFunctionProcessor() = default;
};

#endif   // CFUNCTION_PROCESSOR_HPP
