#ifndef SYMBOL_TABLE_HPP
#define SYMBOL_TABLE_HPP

#include <PugsMacros.hpp>

#include <ASTNodeDataType.hpp>
#include <ASTNodeDataVariant.hpp>

#include <pegtl/position.hpp>

#include <iostream>

#include <CFunctionEmbedderTable.hpp>
#include <FunctionTable.hpp>

class SymbolTable
{
 public:
  class Attributes
  {
   private:
    bool m_is_initialized{false};
    ASTNodeDataType m_data_type{ASTNodeDataType::undefined_t};
    ASTNodeDataVariant m_value;

    TAO_PEGTL_NAMESPACE::position m_position;

   public:
    auto&
    value()
    {
      return m_value;
    }

    const auto&
    value() const
    {
      return m_value;
    }

    const bool&
    isInitialized() const
    {
      return m_is_initialized;
    }

    void
    setIsInitialized()
    {
      m_is_initialized = true;
    }

    const ASTNodeDataType&
    dataType() const
    {
      return m_data_type;
    }

    auto
    position() const
    {
      return m_position;
    }

    void
    setDataType(const ASTNodeDataType& data_type)
    {
      m_data_type = data_type;
    }

    friend std::ostream&
    operator<<(std::ostream& os, const Attributes& attributes)
    {
      if (attributes.m_data_type == ASTNodeDataType::function_t) {
        os << "function_id:";
      }
      std::visit(
        [&](const auto& value) {
          using T = std::decay_t<decltype(value)>;
          if constexpr (std::is_same_v<T, std::monostate>) {
            os << "--";
          } else {
            os << value;
          }
        },
        attributes.m_value);

      return os;
    }

    Attributes(const TAO_PEGTL_NAMESPACE::position& position) : m_position(position) {}

    Attributes(const Attributes&) = default;
  };

  class Symbol
  {
   private:
    std::string m_name;
    Attributes m_attributes;

   public:
    PUGS_INLINE
    const std::string&
    name() const
    {
      return m_name;
    }

    PUGS_INLINE
    const Attributes&
    attributes() const
    {
      return m_attributes;
    }

    PUGS_INLINE
    Attributes&
    attributes()
    {
      return m_attributes;
    }

    Symbol(const std::string& name, const Attributes& attributes) : m_name(name), m_attributes(attributes) {}

    Symbol& operator=(Symbol&&) = default;
    Symbol& operator=(const Symbol&) = default;

    Symbol(const Symbol&) = default;
    Symbol(Symbol&&)      = default;
    ~Symbol()             = default;
  };

 private:
  std::vector<Symbol> m_symbol_list;
  std::shared_ptr<SymbolTable> m_parent_table;
  std::shared_ptr<FunctionTable> m_function_table;
  std::shared_ptr<CFunctionEmbedderTable> m_c_function_embedder_table;

 public:
  const FunctionTable&
  functionTable() const
  {
    Assert(m_function_table);
    return *m_function_table;
  }

  FunctionTable&
  functionTable()
  {
    Assert(m_function_table);
    return *m_function_table;
  }

  const CFunctionEmbedderTable&
  cFunctionEbedderTable() const
  {
    Assert(m_c_function_embedder_table);
    return *m_c_function_embedder_table;
  }

  CFunctionEmbedderTable&
  cFunctionEbedderTable()
  {
    Assert(m_c_function_embedder_table);
    return *m_c_function_embedder_table;
  }

  friend std::ostream&
  operator<<(std::ostream& os, const SymbolTable& symbol_table)
  {
    os << "-- Symbol table state -- parent : " << symbol_table.m_parent_table.get() << "\n";
    for (auto i_symbol : symbol_table.m_symbol_list) {
      if (i_symbol.attributes().dataType() != ASTNodeDataType::c_function_t) {
        os << ' ' << i_symbol.name() << ": " << std::boolalpha << i_symbol.attributes() << '\n';
      }
    }
    os << "------------------------\n";
    return os;
  }

  auto
  find(const std::string& symbol, const TAO_PEGTL_NAMESPACE::position& use_position)
  {
    auto i_symbol = m_symbol_list.end();

    for (auto i_stored_symbol = m_symbol_list.begin(); i_stored_symbol != m_symbol_list.end(); ++i_stored_symbol) {
      if (i_stored_symbol->name() == symbol) {
        i_symbol = i_stored_symbol;
        break;
      }
    }

    if (i_symbol != m_symbol_list.end() and (use_position.byte >= i_symbol->attributes().position().byte)) {
      return std::make_pair(i_symbol, true);
    } else {
      if (m_parent_table) {
        return m_parent_table->find(symbol, use_position);
      } else {
        return std::make_pair(i_symbol, false);
      }
    }
  }

  auto
  add(const std::string& symbol_name, const TAO_PEGTL_NAMESPACE::position& symbol_position)
  {
    for (auto i_stored_symbol = m_symbol_list.begin(); i_stored_symbol != m_symbol_list.end(); ++i_stored_symbol) {
      if (i_stored_symbol->name() == symbol_name) {
        return std::make_pair(i_stored_symbol, false);
      }
    }
    return std::make_pair(m_symbol_list.emplace(m_symbol_list.end(), Symbol{symbol_name, Attributes(symbol_position)}),
                          true);
  }

  SymbolTable(const std::shared_ptr<SymbolTable>& parent_table)
    : m_parent_table(parent_table),
      m_function_table(parent_table->m_function_table),
      m_c_function_embedder_table(parent_table->m_c_function_embedder_table)
  {
    ;
  }

  SymbolTable()
    : m_parent_table(nullptr),
      m_function_table(std::make_shared<FunctionTable>()),
      m_c_function_embedder_table(std::make_shared<CFunctionEmbedderTable>())
  {
    ;
  }
};

#endif   // SYMBOL_TABLE_HPP
