#include <ASTNodeFunctionExpressionBuilder.hpp>
#include <PEGGrammar.hpp>

#include <FunctionTable.hpp>
#include <SymbolTable.hpp>

#include <node_processor/FunctionProcessor.hpp>

template <typename SymbolType>
PUGS_INLINE std::unique_ptr<INodeProcessor>
ASTNodeFunctionExpressionBuilder::_getArgumentProcessor(ASTNode& argument_node, SymbolType& parameter_symbol)
{
  auto get_function_argument_processor_for_parameter_type =
    [&](const auto& argument_v) -> std::unique_ptr<INodeProcessor> {
    using ArgumentT = std::decay_t<decltype(argument_v)>;
    switch (parameter_symbol.attributes().dataType()) {
    case ASTNodeDataType::bool_t: {
      return std::make_unique<FunctionArgumentProcessor<ArgumentT, bool>>(argument_node, parameter_symbol);
    }
    case ASTNodeDataType::unsigned_int_t: {
      return std::make_unique<FunctionArgumentProcessor<ArgumentT, uint64_t>>(argument_node, parameter_symbol);
    }
    case ASTNodeDataType::int_t: {
      return std::make_unique<FunctionArgumentProcessor<ArgumentT, int64_t>>(argument_node, parameter_symbol);
    }
    case ASTNodeDataType::double_t: {
      return std::make_unique<FunctionArgumentProcessor<ArgumentT, double>>(argument_node, parameter_symbol);
    }
      // LCOV_EXCL_START
    default: {
      throw parse_error("unexpected error: undefined parameter type for function", std::vector{argument_node.begin()});
    }
      // LCOV_EXCL_STOP
    }
  };

  auto get_function_argument_processor_for_argument_type = [&]() {
    switch (argument_node.m_data_type) {
    case ASTNodeDataType::bool_t: {
      return get_function_argument_processor_for_parameter_type(bool{});
    }
    case ASTNodeDataType::unsigned_int_t: {
      return get_function_argument_processor_for_parameter_type(uint64_t{});
    }
    case ASTNodeDataType::int_t: {
      return get_function_argument_processor_for_parameter_type(int64_t{});
    }
    case ASTNodeDataType::double_t: {
      return get_function_argument_processor_for_parameter_type(double{});
    }
      // LCOV_EXCL_START
    default: {
      throw parse_error("unexpected error: undefined argument type for function", std::vector{argument_node.begin()});
    }
      // LCOV_EXCL_STOP
    }
  };

  return get_function_argument_processor_for_argument_type();
}

PUGS_INLINE
void
ASTNodeFunctionExpressionBuilder::_storeArgumentProcessor(ASTNode& parameter_variable,
                                                          ASTNode& argument_node,
                                                          FunctionProcessor& function_processor)
{
  Assert(parameter_variable.is_type<language::name>(), "unexpected parameter type!");

  auto [i_parameter_symbol, found] =
    parameter_variable.m_symbol_table->find(parameter_variable.string(), parameter_variable.begin());
  Assert(found);

  function_processor.addArgumentProcessor(this->_getArgumentProcessor(argument_node, *i_parameter_symbol));
}

PUGS_INLINE
void
ASTNodeFunctionExpressionBuilder::_buildArgumentProcessors(FunctionDescriptor& function_descriptor,
                                                           ASTNode& node,
                                                           FunctionProcessor& function_processor)
{
  const ASTNode& definition_node = function_descriptor.definitionNode();
  ASTNode& parameter_variables   = *definition_node.children[0];

  ASTNode& argument_nodes = *node.children[1];

  const size_t arguments_number =
    argument_nodes.is_type<language::function_argument_list>() ? argument_nodes.children.size() : 1;
  const size_t parameters_number =
    parameter_variables.is_type<language::name_list>() ? parameter_variables.children.size() : 1;

  if (arguments_number != parameters_number) {
    std::ostringstream error_message;
    error_message << "bad number of arguments: expecting " << rang::fgB::yellow << parameters_number
                  << rang::style::reset << ", provided " << rang::fgB::yellow << arguments_number << rang::style::reset;
    throw parse_error(error_message.str(), argument_nodes.begin());
  }

  if (arguments_number > 1) {
    for (size_t i = 0; i < arguments_number; ++i) {
      ASTNode& parameter_variable = *parameter_variables.children[i];
      ASTNode& argument_node      = *argument_nodes.children[i];
      this->_storeArgumentProcessor(parameter_variable, argument_node, function_processor);
    }
  } else {
    this->_storeArgumentProcessor(parameter_variables, argument_nodes, function_processor);
  }
}

PUGS_INLINE
std::unique_ptr<INodeProcessor>
ASTNodeFunctionExpressionBuilder::_getFunctionProcessor(const ASTNodeDataType expression_value_type,
                                                        const ASTNodeDataType return_value_type,
                                                        ASTNode& node)
{
  auto get_function_processor_for_expression_value = [&](const auto& return_v) -> std::unique_ptr<INodeProcessor> {
    using ReturnT = std::decay_t<decltype(return_v)>;
    switch (expression_value_type) {
    case ASTNodeDataType::bool_t: {
      return std::make_unique<FunctionExpressionProcessor<ReturnT, bool>>(node);
    }
    case ASTNodeDataType::unsigned_int_t: {
      return std::make_unique<FunctionExpressionProcessor<ReturnT, uint64_t>>(node);
    }
    case ASTNodeDataType::int_t: {
      return std::make_unique<FunctionExpressionProcessor<ReturnT, int64_t>>(node);
    }
    case ASTNodeDataType::double_t: {
      return std::make_unique<FunctionExpressionProcessor<ReturnT, double>>(node);
    }
      // LCOV_EXCL_START
    default: {
      throw parse_error("unexpected error: undefined expression value type for function",
                        std::vector{node.children[1]->begin()});
    }
      // LCOV_EXCL_STOP
    }
  };

  auto get_function_processor_for_value = [&]() {
    switch (return_value_type) {
    case ASTNodeDataType::bool_t: {
      return get_function_processor_for_expression_value(bool{});
    }
    case ASTNodeDataType::unsigned_int_t: {
      return get_function_processor_for_expression_value(uint64_t{});
    }
    case ASTNodeDataType::int_t: {
      return get_function_processor_for_expression_value(int64_t{});
    }
    case ASTNodeDataType::double_t: {
      return get_function_processor_for_expression_value(double{});
    }
      // LCOV_EXCL_START
    default: {
      throw parse_error("unexpected error: undefined return type for function", std::vector{node.begin()});
    }
      // LCOV_EXCL_STOP
    }
  };

  return get_function_processor_for_value();
}

ASTNodeFunctionExpressionBuilder::ASTNodeFunctionExpressionBuilder(ASTNode& node)
{
  auto [i_function_symbol, found] = node.m_symbol_table->find(node.children[0]->string(), node.begin());
  Assert(found);
  Assert(i_function_symbol->attributes().dataType() == ASTNodeDataType::function_t);

  uint64_t function_id = std::get<uint64_t>(i_function_symbol->attributes().value());

  FunctionDescriptor& function_descriptor = node.m_symbol_table->functionTable()[function_id];

  if (function_descriptor.definitionNode().children[1]->is_type<language::expression_list>()) {
    // LCOV_EXCL_START
    throw parse_error("unexpected error: function expression list is not implemented yet",
                      std::vector{function_descriptor.definitionNode().children[1]->begin()});
    // LCOV_EXCL_STOP
  }

  std::unique_ptr function_processor = std::make_unique<FunctionProcessor>();

  this->_buildArgumentProcessors(function_descriptor, node, *function_processor);

  const ASTNodeDataType expression_value_type = function_descriptor.definitionNode().children[1]->m_data_type;
  const ASTNodeDataType return_value_type     = node.m_data_type;
  function_processor->addFunctionExpressionProcessor(
    this->_getFunctionProcessor(expression_value_type, return_value_type, node));

  node.m_node_processor = std::move(function_processor);
}
