#include <ASTNodeCFunctionExpressionBuilder.hpp>
#include <PEGGrammar.hpp>

#include <SymbolTable.hpp>

#include <node_processor/CFunctionProcessor.hpp>

PUGS_INLINE std::unique_ptr<INodeProcessor>
ASTNodeCFunctionExpressionBuilder::_getArgumentProcessor(ASTNode& argument_node,
                                                         const ASTNodeDataType& argument_type,
                                                         ASTNodeDataVariant& argument_value)
{
  auto get_function_argument_processor_for_parameter_type =
    [&](const auto& argument_v) -> std::unique_ptr<INodeProcessor> {
    using ArgumentT = std::decay_t<decltype(argument_v)>;
    switch (argument_type) {
    case ASTNodeDataType::bool_t: {
      return std::make_unique<CFunctionArgumentProcessor<ArgumentT, bool>>(argument_node, argument_value);
    }
    case ASTNodeDataType::unsigned_int_t: {
      return std::make_unique<CFunctionArgumentProcessor<ArgumentT, uint64_t>>(argument_node, argument_value);
    }
    case ASTNodeDataType::int_t: {
      return std::make_unique<CFunctionArgumentProcessor<ArgumentT, int64_t>>(argument_node, argument_value);
    }
    case ASTNodeDataType::double_t: {
      return std::make_unique<CFunctionArgumentProcessor<ArgumentT, double>>(argument_node, argument_value);
    }
    default: {
      throw parse_error("unexpected error: undefined parameter type for function", std::vector{argument_node.begin()});
    }
    }
  };

  auto get_function_argument_processor_for_argument_type = [&]() {
    switch (argument_node.m_data_type) {
    case ASTNodeDataType::bool_t: {
      return get_function_argument_processor_for_parameter_type(bool{});
    }
    case ASTNodeDataType::unsigned_int_t: {
      return get_function_argument_processor_for_parameter_type(uint64_t{});
    }
    case ASTNodeDataType::int_t: {
      return get_function_argument_processor_for_parameter_type(int64_t{});
    }
    case ASTNodeDataType::double_t: {
      return get_function_argument_processor_for_parameter_type(double{});
    }
    default: {
      throw parse_error("invalid argument type for function", std::vector{argument_node.begin()});
    }
    }
  };

  return get_function_argument_processor_for_argument_type();
}

PUGS_INLINE
void
ASTNodeCFunctionExpressionBuilder::_storeArgumentProcessor(const size_t argument_number,
                                                           ASTNode& argument_node,
                                                           const std::vector<ASTNodeDataType>& argument_type_list,
                                                           CFunctionProcessor& c_function_processor)
{
  auto& argument_values = c_function_processor.argumentValues();

  c_function_processor.addArgumentProcessor(
    this->_getArgumentProcessor(argument_node, argument_type_list[argument_number], argument_values[argument_number]));
}

PUGS_INLINE
void
ASTNodeCFunctionExpressionBuilder::_buildArgumentProcessors(ASTNode& node,
                                                            const std::vector<ASTNodeDataType>& argument_type_list,
                                                            CFunctionProcessor& c_function_processor)
{
  ASTNode& argument_nodes = *node.children[1];

  const size_t arguments_number =
    argument_nodes.is_type<language::function_argument_list>() ? argument_nodes.children.size() : 1;

  c_function_processor.setNumberOfArguments(arguments_number);

  const size_t parameters_number = argument_type_list.size();

  if (arguments_number != parameters_number) {
    std::ostringstream error_message;
    error_message << "bad number of arguments: expecting " << rang::fgB::yellow << parameters_number
                  << rang::style::reset << ", provided " << rang::fgB::yellow << arguments_number << rang::style::reset;
    throw parse_error(error_message.str(), argument_nodes.begin());
  }

  if (arguments_number > 1) {
    for (size_t i = 0; i < arguments_number; ++i) {
      ASTNode& argument_node = *argument_nodes.children[i];
      this->_storeArgumentProcessor(i, argument_node, argument_type_list, c_function_processor);
    }
  } else {
    this->_storeArgumentProcessor(0, argument_nodes, argument_type_list, c_function_processor);
  }
}

ASTNodeCFunctionExpressionBuilder::ASTNodeCFunctionExpressionBuilder(ASTNode& node)
{
  auto [i_function_symbol, found] = node.m_symbol_table->find(node.children[0]->string(), node.begin());
  Assert(found);
  Assert(i_function_symbol->attributes().dataType() == ASTNodeDataType::c_function_t);

  uint64_t c_function_id = std::get<uint64_t>(i_function_symbol->attributes().value());

  CFunctionEmbedderTable& c_function_embedder_table = node.m_symbol_table->cFunctionEbedderTable();
  std::shared_ptr c_function_embedder               = c_function_embedder_table[c_function_id];

  std::vector<ASTNodeDataType> c_function_argument_type_list = c_function_embedder->getArgumentDataTypes();

  std::unique_ptr c_function_processor = std::make_unique<CFunctionProcessor>();

  this->_buildArgumentProcessors(node, c_function_argument_type_list, *c_function_processor);

  c_function_processor->setFunctionExpressionProcessor(
    std::make_unique<CFunctionExpressionProcessor>(node, c_function_embedder, c_function_processor->argumentValues()));

  node.m_node_processor = std::move(c_function_processor);
}
