#include <ASTNodeExpressionBuilder.hpp>

#include <ASTNodeAffectationExpressionBuilder.hpp>
#include <ASTNodeBinaryOperatorExpressionBuilder.hpp>
#include <ASTNodeIncDecExpressionBuilder.hpp>
#include <ASTNodeUnaryOperatorExpressionBuilder.hpp>

#include <node_processor/ASTNodeListProcessor.hpp>
#include <node_processor/BreakProcessor.hpp>
#include <node_processor/ContinueProcessor.hpp>
#include <node_processor/DoWhileProcessor.hpp>
#include <node_processor/FakeProcessor.hpp>
#include <node_processor/ForProcessor.hpp>
#include <node_processor/IfProcessor.hpp>
#include <node_processor/NameProcessor.hpp>
#include <node_processor/OStreamProcessor.hpp>
#include <node_processor/WhileProcessor.hpp>

#include <PEGGrammar.hpp>

void
ASTNodeExpressionBuilder::_buildExpression(ASTNode& n)
{
  if (n.is<language::block>()) {
    n.m_node_processor = std::make_unique<ASTNodeListProcessor>(n);
  } else if ((n.is<language::eq_op>() or n.is<language::multiplyeq_op>() or n.is<language::divideeq_op>() or
              n.is<language::pluseq_op>() or n.is<language::minuseq_op>())) {
    ASTNodeAffectationExpressionBuilder{n};

  } else if (n.is<language::let_declaration>()) {
    std::cerr << rang::fgB::red << "\"let expression\" is not defined correctly (declaration should be removed)"
              << rang::style::reset << '\n';
    n.m_node_processor = std::make_unique<FakeProcessor>();
    return;

  } else if (n.is<language::function_evaluation>()) {
    std::cerr << rang::fgB::red << "\"function evaluation\" is not defined yet" << rang::style::reset << '\n';
    n.m_node_processor = std::make_unique<FakeProcessor>();
    return;

  } else if (n.is<language::real>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();
  } else if (n.is<language::integer>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();
  } else if (n.is<language::literal>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();
  } else if (n.is<language::true_kw>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();
  } else if (n.is<language::false_kw>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();

  } else if (n.is<language::name>()) {
    n.m_node_processor = std::make_unique<NameProcessor>(n);

  } else if (n.is<language::unary_minus>() or n.is<language::unary_not>()) {
    ASTNodeUnaryOperatorExpressionBuilder{n};

  } else if (n.is<language::unary_minusminus>() or n.is<language::unary_plusplus>() or
             n.is<language::post_minusminus>() or n.is<language::post_plusplus>()) {
    ASTNodeIncDecExpressionBuilder{n};

  } else if (n.is<language::multiply_op>() or n.is<language::divide_op>() or n.is<language::plus_op>() or
             n.is<language::minus_op>() or n.is<language::or_op>() or n.is<language::and_op>() or
             n.is<language::xor_op>() or n.is<language::greater_op>() or n.is<language::greater_or_eq_op>() or
             n.is<language::lesser_op>() or n.is<language::lesser_or_eq_op>() or n.is<language::eqeq_op>() or
             n.is<language::not_eq_op>()) {
    ASTNodeBinaryOperatorExpressionBuilder{n};

  } else if (n.is<language::cout_kw>()) {
    n.m_node_processor = std::make_unique<OStreamProcessor>(n, std::cout);
  } else if (n.is<language::cerr_kw>()) {
    n.m_node_processor = std::make_unique<OStreamProcessor>(n, std::cerr);
  } else if (n.is<language::clog_kw>()) {
    n.m_node_processor = std::make_unique<OStreamProcessor>(n, std::clog);
  } else if (n.is<language::if_statement>()) {
    n.m_node_processor = std::make_unique<IfProcessor>(n);
  } else if (n.is<language::statement_block>()) {
    n.m_node_processor = std::make_unique<ASTNodeListProcessor>(n);
  } else if (n.is<language::do_while_statement>()) {
    n.m_node_processor = std::make_unique<DoWhileProcessor>(n);
  } else if (n.is<language::while_statement>()) {
    n.m_node_processor = std::make_unique<WhileProcessor>(n);
  } else if (n.is<language::for_statement>()) {
    n.m_node_processor = std::make_unique<ForProcessor>(n);
  } else if (n.is<language::for_statement_block>()) {
    n.m_node_processor = std::make_unique<ASTNodeListProcessor>(n);
  } else if (n.is<language::for_init>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();
  } else if (n.is<language::for_post>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();
  } else if (n.is<language::for_test>()) {
    n.m_node_processor = std::make_unique<FakeProcessor>();
  } else if (n.is<language::break_kw>()) {
    n.m_node_processor = std::make_unique<BreakProcessor>();
  } else if (n.is<language::continue_kw>()) {
    n.m_node_processor = std::make_unique<ContinueProcessor>();
  } else {
    std::ostringstream error_message;
    error_message << "undefined node type '" << rang::fgB::red << n.name() << rang::fg::reset << "'";
    throw parse_error{error_message.str(), std::vector{n.begin()}};
  }

  for (auto& child : n.children) {
    this->_buildExpression(*child);
  }
}

ASTNodeExpressionBuilder::ASTNodeExpressionBuilder(ASTNode& n)
{
  Assert(n.is_root());
  n.m_node_processor = std::make_unique<ASTNodeListProcessor>(n);
  for (auto& child : n.children) {
    this->_buildExpression(*child);
  }
  std::cout << " - build node types\n";
}
