#include <ASTNodeAffectationExpressionBuilder.hpp>
#include <PEGGrammar.hpp>

#include <node_processor/AffectationProcessor.hpp>
#include <node_processor/AffectationToStringProcessor.hpp>

ASTNodeAffectationExpressionBuilder::ASTNodeAffectationExpressionBuilder(ASTNode& n)
{
  auto set_affectation_processor = [](ASTNode& n, const auto& operator_v) {
    auto set_affectation_processor_for_data = [&](const auto& value, const ASTNodeDataType& data_type) {
      using OperatorT = std::decay_t<decltype(operator_v)>;
      using ValueT    = std::decay_t<decltype(value)>;
      switch (data_type) {
      case ASTNodeDataType::bool_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, bool>>(n);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, uint64_t>>(n);
        break;
      }
      case ASTNodeDataType::int_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, int64_t>>(n);
        break;
      }
      case ASTNodeDataType::double_t: {
        n.m_node_processor = std::make_unique<AffectationProcessor<OperatorT, ValueT, double>>(n);
        break;
      }
      default: {
        throw parse_error("unexpected error: undefined operand type for affectation",
                          std::vector{n.children[1]->begin()});
      }
      }
    };

    auto set_affectation_processor_for_string_data = [&](const ASTNodeDataType& data_type) {
      using OperatorT = std::decay_t<decltype(operator_v)>;

      if constexpr (std::is_same_v<OperatorT, language::eq_op> or std::is_same_v<OperatorT, language::pluseq_op>) {
        switch (data_type) {
        case ASTNodeDataType::bool_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, bool>>(n);
          break;
        }
        case ASTNodeDataType::unsigned_int_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, uint64_t>>(n);
          break;
        }
        case ASTNodeDataType::int_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, int64_t>>(n);
          break;
        }
        case ASTNodeDataType::double_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, double>>(n);
          break;
        }
        case ASTNodeDataType::string_t: {
          n.m_node_processor = std::make_unique<AffectationToStringProcessor<OperatorT, std::string>>(n);
          break;
        }
        default: {
          throw parse_error("unexpected error: undefined operand type for string affectation",
                            std::vector{n.children[1]->begin()});
        }
        }
      } else {
        throw parse_error("unexpected error: undefined operator type", std::vector{n.children[0]->begin()});
      }
    };

    auto set_affectation_processor_for_value = [&](const ASTNodeDataType& value_type) {
      const ASTNodeDataType data_type = n.children[1]->m_data_type;
      switch (value_type) {
      case ASTNodeDataType::bool_t: {
        set_affectation_processor_for_data(bool{}, data_type);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        set_affectation_processor_for_data(uint64_t{}, data_type);
        break;
      }
      case ASTNodeDataType::int_t: {
        set_affectation_processor_for_data(int64_t{}, data_type);
        break;
      }
      case ASTNodeDataType::double_t: {
        set_affectation_processor_for_data(double{}, data_type);
        break;
      }
      case ASTNodeDataType::string_t: {
        set_affectation_processor_for_string_data(data_type);
        break;
      }
      default: {
        throw parse_error("unexpected error: undefined value type for affectation", std::vector{n.begin()});
      }
      }
    };

    set_affectation_processor_for_value(n.m_data_type);
  };

  if (n.is<language::eq_op>()) {
    set_affectation_processor(n, language::eq_op{});
  } else if (n.is<language::multiplyeq_op>()) {
    set_affectation_processor(n, language::multiplyeq_op{});
  } else if (n.is<language::divideeq_op>()) {
    set_affectation_processor(n, language::divideeq_op{});
  } else if (n.is<language::pluseq_op>()) {
    set_affectation_processor(n, language::pluseq_op{});
  } else if (n.is<language::minuseq_op>()) {
    set_affectation_processor(n, language::minuseq_op{});
  } else {
    throw parse_error("unexpected error: undefined affectation operator", std::vector{n.begin()});
  }
}
