#ifndef INTERPOLATE_ITEM_ARRAY_HPP
#define INTERPOLATE_ITEM_ARRAY_HPP

#include <language/utils/InterpolateItemValue.hpp>
#include <mesh/ItemArray.hpp>
#include <mesh/ItemType.hpp>

template <typename T>
class InterpolateItemArray;
template <typename OutputType, typename InputType>
class InterpolateItemArray<OutputType(InputType)>
{
  static constexpr size_t Dimension = OutputType::Dimension;

 public:
  template <ItemType item_type>
  PUGS_INLINE static ItemArray<OutputType, item_type>
  interpolate(const std::vector<FunctionSymbolId>& function_symbol_id_list,
              const ItemValue<const InputType, item_type>& position)
  {
    ItemArray<OutputType, item_type> item_array{*position.connectivity_ptr(), function_symbol_id_list.size()};

    for (size_t i_function_symbol = 0; i_function_symbol < function_symbol_id_list.size(); ++i_function_symbol) {
      const FunctionSymbolId& function_symbol_id = function_symbol_id_list[i_function_symbol];
      ItemValue<OutputType, item_type> item_value =
        InterpolateItemValue<OutputType(InputType)>::interpolate(function_symbol_id, position);
      parallel_for(
        item_value.numberOfItems(),
        PUGS_LAMBDA(ItemIdT<item_type> item_id) { item_array[item_id][i_function_symbol] = item_value[item_id]; });
    }

    return item_array;
  }

  template <ItemType item_type>
  PUGS_INLINE static Table<OutputType>
  interpolate(const std::vector<FunctionSymbolId>& function_symbol_id_list,
              const ItemValue<const InputType, item_type>& position,
              const Array<const ItemIdT<item_type>>& list_of_items)
  {
    Table<OutputType> table{list_of_items.size(), function_symbol_id_list.size()};

    for (size_t i_function_symbol = 0; i_function_symbol < function_symbol_id_list.size(); ++i_function_symbol) {
      const FunctionSymbolId& function_symbol_id = function_symbol_id_list[i_function_symbol];
      Array<OutputType> array =
        InterpolateItemValue<OutputType(InputType)>::interpolate(function_symbol_id, position, list_of_items);

      parallel_for(
        array.size(), PUGS_LAMBDA(size_t i) { table[i][i_function_symbol] = array[i]; });
    }

    return table;
  }

  template <ItemType item_type>
  PUGS_INLINE static Table<OutputType>
  interpolate(const std::vector<FunctionSymbolId>& function_symbol_id_list,
              const ItemValue<const InputType, item_type>& position,
              const Array<ItemIdT<item_type>>& list_of_items)
  {
    return interpolate(function_symbol_id_list, position, Array<const ItemIdT<item_type>>{list_of_items});
  }
};

#endif   // INTERPOLATE_ITEM_ARRAY_HPP
