#ifndef OPERATOR_REPOSITORY_HPP
#define OPERATOR_REPOSITORY_HPP

#include <language/node_processor/INodeProcessor.hpp>
#include <language/utils/ASTNodeDataType.hpp>
#include <language/utils/AffectationMangler.hpp>
#include <language/utils/IAffectationProcessorBuilder.hpp>
#include <language/utils/IUnaryOperatorProcessorBuilder.hpp>
#include <language/utils/UnaryOperatorMangler.hpp>

#include <utils/Exceptions.hpp>

#include <optional>

class OperatorRepository
{
 private:
  std::unordered_map<std::string, std::shared_ptr<const IAffectationProcessorBuilder>> m_affectation_builder_list;
  std::unordered_map<std::string, std::shared_ptr<const IUnaryOperatorProcessorBuilder>> m_unary_operator_builder_list;

  void _initialize();

 public:
  void reset();

  template <typename OperatorTypeT, typename AffectationProcessorBuilderT>
  void
  addAffectation(const ASTNodeDataType& lhs,
                 const ASTNodeDataType& rhs,
                 const std::shared_ptr<AffectationProcessorBuilderT>& processor_builder)
  {
    const std::string affectation_type_name = affectationMangler<OperatorTypeT>(lhs, rhs);
    if (not m_affectation_builder_list.try_emplace(affectation_type_name, processor_builder).second) {
      throw UnexpectedError(affectation_type_name + " has already an entry");
    }
  }

  template <typename OperatorTypeT, typename UnaryProcessorBuilderT>
  void
  addUnaryOperator(const ASTNodeDataType& operand, const std::shared_ptr<UnaryProcessorBuilderT>& processor_builder)
  {
    const std::string unary_operator_type_name = unaryOperatorMangler<OperatorTypeT>(operand);
    if (not m_unary_operator_builder_list.try_emplace(unary_operator_type_name, processor_builder).second) {
      throw UnexpectedError(unary_operator_type_name + " has already an entry");
    }
  }

  std::optional<std::shared_ptr<const IAffectationProcessorBuilder>>
  getAffectationProcessorBuilder(const std::string& affectation_name) const
  {
    auto&& processor_builder = m_affectation_builder_list.find(affectation_name);
    if (processor_builder != m_affectation_builder_list.end()) {
      return processor_builder->second;
    }
    return {};
  }

  std::optional<std::shared_ptr<const IUnaryOperatorProcessorBuilder>>
  getUnaryProcessorBuilder(const std::string& affectation_name) const
  {
    auto&& processor_builder = m_unary_operator_builder_list.find(affectation_name);
    if (processor_builder != m_unary_operator_builder_list.end()) {
      return processor_builder->second;
    }
    return {};
  }

  static void create();

  PUGS_INLINE
  static OperatorRepository&
  instance()
  {
    Assert(m_instance != nullptr);
    return *m_instance;
  }

  static void destroy();

 private:
  static OperatorRepository* m_instance;

  OperatorRepository() = default;

  ~OperatorRepository() = default;
};

#endif   // OPERATOR_REPOSITORY_HPP
