#include <language/ast/ASTNodeAffectationExpressionBuilder.hpp>

#include <algebra/TinyVector.hpp>
#include <language/PEGGrammar.hpp>
#include <language/ast/ASTNodeNaturalConversionChecker.hpp>
#include <language/node_processor/INodeProcessor.hpp>
#include <language/utils/AffectationMangler.hpp>
#include <language/utils/OperatorRepository.hpp>
#include <language/utils/ParseError.hpp>
#include <utils/Exceptions.hpp>

ASTNodeAffectationExpressionBuilder::ASTNodeAffectationExpressionBuilder(ASTNode& n)
{
  const ASTNodeDataType& target_data_type = n.children[0]->m_data_type;
  const ASTNodeDataType& source_data_type = n.children[1]->m_data_type;

  const std::string affectation_name = [&] {
    if (n.is_type<language::eq_op>()) {
      return affectationMangler<language::eq_op>(target_data_type, source_data_type);
    } else if (n.is_type<language::multiplyeq_op>()) {
      return affectationMangler<language::multiplyeq_op>(target_data_type, source_data_type);
    } else if (n.is_type<language::divideeq_op>()) {
      return affectationMangler<language::divideeq_op>(target_data_type, source_data_type);
    } else if (n.is_type<language::pluseq_op>()) {
      return affectationMangler<language::pluseq_op>(target_data_type, source_data_type);
    } else if (n.is_type<language::minuseq_op>()) {
      return affectationMangler<language::minuseq_op>(target_data_type, source_data_type);
    } else {
      throw ParseError("unexpected error: undefined affectation operator", std::vector{n.begin()});
    }
  }();

  // Special treatment dedicated to R^1 to be able to initialize them
  if (((target_data_type != source_data_type) and (target_data_type == ASTNodeDataType::vector_t) and
       (target_data_type.dimension() == 1)) or
      // Special treatment for R^d vectors and operator *=
      ((target_data_type == ASTNodeDataType::vector_t) and (source_data_type != ASTNodeDataType::vector_t) and
       n.is_type<language::multiplyeq_op>())) {
    ASTNodeNaturalConversionChecker{*n.children[1], ASTNodeDataType::build<ASTNodeDataType::double_t>()};
  } else {
    ASTNodeNaturalConversionChecker{*n.children[1], target_data_type};
  }

  const auto& optional_processor_builder =
    OperatorRepository::instance().getAffectationProcessorBuilder(affectation_name);

  if (optional_processor_builder.has_value()) {
    n.m_node_processor = optional_processor_builder.value()->getNodeProcessor(n);
  } else {
    std::ostringstream error_message;
    error_message << "undefined affectation type: ";
    error_message << rang::fgB::red << affectation_name << rang::fg::reset;

    throw ParseError(error_message.str(), std::vector{n.children[0]->begin()});
  }
}
