#include <language/ASTNodeUnaryOperatorExpressionBuilder.hpp>

#include <language/PEGGrammar.hpp>
#include <language/node_processor/UnaryExpressionProcessor.hpp>

ASTNodeUnaryOperatorExpressionBuilder::ASTNodeUnaryOperatorExpressionBuilder(ASTNode& n)
{
  auto set_unary_operator_processor = [](ASTNode& n, const auto& operator_v) {
    using OperatorT = std::decay_t<decltype(operator_v)>;

    auto set_unary_operator_processor_for_data = [&](const auto& value, const ASTNodeDataType& data_type) {
      using ValueT = std::decay_t<decltype(value)>;
      switch (data_type) {
      case ASTNodeDataType::bool_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, bool>>(n);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, uint64_t>>(n);
        break;
      }
      case ASTNodeDataType::int_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, int64_t>>(n);
        break;
      }
      case ASTNodeDataType::double_t: {
        n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, double>>(n);
        break;
      }
      default: {
        throw parse_error("unexpected error: invalid operand type for unary operator",
                          std::vector{n.children[0]->begin()});
      }
      }
    };

    auto set_unary_operator_processor_for_value = [&](const ASTNodeDataType& value_type) {
      const ASTNodeDataType data_type = n.children[0]->m_data_type;
      switch (value_type) {
      case ASTNodeDataType::bool_t: {
        set_unary_operator_processor_for_data(bool{}, data_type);
        break;
      }
      case ASTNodeDataType::int_t: {
        set_unary_operator_processor_for_data(int64_t{}, data_type);
        break;
      }
      case ASTNodeDataType::double_t: {
        set_unary_operator_processor_for_data(double{}, data_type);
        break;
      }
      case ASTNodeDataType::vector_t: {
        if constexpr (std::is_same_v<OperatorT, language::unary_minus>) {
          switch (data_type.dimension()) {
          case 1: {
            using ValueT       = TinyVector<1>;
            n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, ValueT>>(n);
            break;
          }
          case 2: {
            using ValueT       = TinyVector<2>;
            n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, ValueT>>(n);
            break;
          }
          case 3: {
            using ValueT       = TinyVector<3>;
            n.m_node_processor = std::make_unique<UnaryExpressionProcessor<OperatorT, ValueT, ValueT>>(n);
            break;
          }
            // LCOV_EXCL_START
          default: {
            throw parse_error("unexpected error: invalid vector dimension", std::vector{n.begin()});
          }
            // LCOV_EXCL_STOP
          }
        } else {
          // LCOV_EXCL_START
          throw parse_error("unexpected error: invalid unary operator for vector data", std::vector{n.begin()});
          // LCOV_EXCL_STOP
        }
        break;
      }
      default: {
        throw parse_error("undefined value type for unary operator", std::vector{n.begin()});
      }
      }
    };

    set_unary_operator_processor_for_value(n.m_data_type);
  };

  if (n.is_type<language::unary_minus>()) {
    set_unary_operator_processor(n, language::unary_minus{});
  } else if (n.is_type<language::unary_not>()) {
    set_unary_operator_processor(n, language::unary_not{});
  } else {
    throw parse_error("unexpected error: undefined unary operator", std::vector{n.begin()});
  }
}
