#include <language/ASTNodeBinaryOperatorExpressionBuilder.hpp>

#include <language/PEGGrammar.hpp>
#include <language/node_processor/BinaryExpressionProcessor.hpp>
#include <language/node_processor/ConcatExpressionProcessor.hpp>

ASTNodeBinaryOperatorExpressionBuilder::ASTNodeBinaryOperatorExpressionBuilder(ASTNode& n)
{
  auto set_binary_operator_processor = [](ASTNode& n, const auto& operator_v) {
    auto set_binary_operator_processor_for_data_b = [&](const auto data_a, const ASTNodeDataType& data_type_b) {
      using OperatorT = std::decay_t<decltype(operator_v)>;
      using DataTA    = std::decay_t<decltype(data_a)>;

      if constexpr (std::is_same_v<DataTA, std::string>) {
        if constexpr (std::is_same_v<OperatorT, language::plus_op>) {
          switch (data_type_b) {
          case ASTNodeDataType::bool_t: {
            n.m_node_processor = std::make_unique<ConcatExpressionProcessor<bool>>(n);
            break;
          }
          case ASTNodeDataType::unsigned_int_t: {
            n.m_node_processor = std::make_unique<ConcatExpressionProcessor<uint64_t>>(n);
            break;
          }
          case ASTNodeDataType::int_t: {
            n.m_node_processor = std::make_unique<ConcatExpressionProcessor<int64_t>>(n);
            break;
          }
          case ASTNodeDataType::double_t: {
            n.m_node_processor = std::make_unique<ConcatExpressionProcessor<double>>(n);
            break;
          }
          case ASTNodeDataType::string_t: {
            n.m_node_processor = std::make_unique<ConcatExpressionProcessor<std::string>>(n);
            break;
          }
          default: {
            throw parse_error("undefined operand type for binary operator", std::vector{n.children[1]->begin()});
          }
          }

        } else if constexpr ((std::is_same_v<OperatorT, language::eqeq_op>) or
                             (std::is_same_v<OperatorT, language::not_eq_op>)) {
          if (data_type_b == ASTNodeDataType::string_t) {
            n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, std::string>>(n);
          } else {
            throw parse_error("undefined operand type for binary operator", std::vector{n.begin()});
          }
        } else {
          throw parse_error("undefined operand type for binary operator", std::vector{n.begin()});
        }
      } else if constexpr (std::is_same_v<DataTA, TinyVector<1>> or std::is_same_v<DataTA, TinyVector<2>> or
                           std::is_same_v<DataTA, TinyVector<3>>) {
        if ((data_type_b == ASTNodeDataType::vector_t)) {
          if constexpr (std::is_same_v<OperatorT, language::plus_op> or std::is_same_v<OperatorT, language::minus_op> or
                        std::is_same_v<OperatorT, language::eqeq_op> or
                        std::is_same_v<OperatorT, language::not_eq_op>) {
            if (data_a.dimension() == data_type_b.dimension()) {
              n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, DataTA>>(n);
            } else {
              throw parse_error("incompatible dimensions of operands", std::vector{n.begin()});
            }
          } else {
            throw parse_error("invalid binary operator", std::vector{n.begin()});
          }
        } else {
          // LCOV_EXCL_START
          throw parse_error("unexpected error: invalid operand type for binary operator",
                            std::vector{n.children[1]->begin()});
          // LCOV_EXCL_STOP
        }
      } else {
        switch (data_type_b) {
        case ASTNodeDataType::bool_t: {
          n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, bool>>(n);
          break;
        }
        case ASTNodeDataType::unsigned_int_t: {
          n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, uint64_t>>(n);
          break;
        }
        case ASTNodeDataType::int_t: {
          n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, int64_t>>(n);
          break;
        }
        case ASTNodeDataType::double_t: {
          n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, double>>(n);
          break;
        }
        case ASTNodeDataType::vector_t: {
          if constexpr (std::is_same_v<OperatorT, language::multiply_op>) {
            switch (data_type_b.dimension()) {
            case 1: {
              n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, TinyVector<1>>>(n);
              break;
            }
            case 2: {
              n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, TinyVector<2>>>(n);
              break;
            }
            case 3: {
              n.m_node_processor = std::make_unique<BinaryExpressionProcessor<OperatorT, DataTA, TinyVector<3>>>(n);
              break;
            }
              // LCOV_EXCL_START
            default: {
              throw parse_error("unexpected error: invalid dimension", std::vector{n.children[0]->begin()});
            }
              // LCOV_EXCL_STOP
            }
            break;
          }
        }
        default: {
          throw parse_error("undefined operand type for binary operator", std::vector{n.children[1]->begin()});
        }
        }
      }
    };

    auto set_binary_operator_processor_for_data_a = [&](const ASTNodeDataType& data_type_a) {
      const ASTNodeDataType data_type_b = n.children[1]->m_data_type;
      switch (data_type_a) {
      case ASTNodeDataType::bool_t: {
        set_binary_operator_processor_for_data_b(bool{}, data_type_b);
        break;
      }
      case ASTNodeDataType::unsigned_int_t: {
        set_binary_operator_processor_for_data_b(uint64_t{}, data_type_b);
        break;
      }
      case ASTNodeDataType::int_t: {
        set_binary_operator_processor_for_data_b(int64_t{}, data_type_b);
        break;
      }
      case ASTNodeDataType::double_t: {
        set_binary_operator_processor_for_data_b(double{}, data_type_b);
        break;
      }
      case ASTNodeDataType::string_t: {
        set_binary_operator_processor_for_data_b(std::string{}, data_type_b);
        break;
      }
      case ASTNodeDataType::vector_t: {
        switch (data_type_a.dimension()) {
        case 1: {
          set_binary_operator_processor_for_data_b(TinyVector<1>{}, data_type_b);
          break;
        }
        case 2: {
          set_binary_operator_processor_for_data_b(TinyVector<2>{}, data_type_b);
          break;
        }
        case 3: {
          set_binary_operator_processor_for_data_b(TinyVector<3>{}, data_type_b);
          break;
        }
          // LCOV_EXCL_START
        default: {
          throw parse_error("unexpected error: invalid dimension", std::vector{n.children[0]->begin()});
        }
          // LCOV_EXCL_STOP
        }
        break;
      }
      default: {
        throw parse_error("undefined operand type for binary operator", std::vector{n.children[0]->begin()});
      }
      }
    };

    set_binary_operator_processor_for_data_a(n.children[0]->m_data_type);
  };

  if (n.is_type<language::multiply_op>()) {
    set_binary_operator_processor(n, language::multiply_op{});
  } else if (n.is_type<language::divide_op>()) {
    set_binary_operator_processor(n, language::divide_op{});
  } else if (n.is_type<language::plus_op>()) {
    set_binary_operator_processor(n, language::plus_op{});
  } else if (n.is_type<language::minus_op>()) {
    set_binary_operator_processor(n, language::minus_op{});

  } else if (n.is_type<language::or_op>()) {
    set_binary_operator_processor(n, language::or_op{});
  } else if (n.is_type<language::and_op>()) {
    set_binary_operator_processor(n, language::and_op{});
  } else if (n.is_type<language::xor_op>()) {
    set_binary_operator_processor(n, language::xor_op{});

  } else if (n.is_type<language::greater_op>()) {
    set_binary_operator_processor(n, language::greater_op{});
  } else if (n.is_type<language::greater_or_eq_op>()) {
    set_binary_operator_processor(n, language::greater_or_eq_op{});
  } else if (n.is_type<language::lesser_op>()) {
    set_binary_operator_processor(n, language::lesser_op{});
  } else if (n.is_type<language::lesser_or_eq_op>()) {
    set_binary_operator_processor(n, language::lesser_or_eq_op{});
  } else if (n.is_type<language::eqeq_op>()) {
    set_binary_operator_processor(n, language::eqeq_op{});
  } else if (n.is_type<language::not_eq_op>()) {
    set_binary_operator_processor(n, language::not_eq_op{});
  } else {
    throw parse_error("unexpected error: undefined binary operator", std::vector{n.begin()});
  }
}
