#ifndef ARRAY_UTILS_HPP
#define ARRAY_UTILS_HPP

#include <PugsMacros.hpp>
#include <PugsUtils.hpp>

#include <Types.hpp>

template <typename DataType, template <typename> typename ArrayT>
std::remove_const_t<DataType>
min(const ArrayT<DataType>& array)
{
  using ArrayType  = ArrayT<DataType>;
  using data_type  = std::remove_const_t<typename ArrayType::data_type>;
  using index_type = typename ArrayType::index_type;

  class ArrayMin
  {
   private:
    const ArrayType m_array;

   public:
    PUGS_INLINE
    operator data_type()
    {
      data_type reduced_value;
      parallel_reduce(m_array.size(), *this, reduced_value);
      return reduced_value;
    }

    PUGS_INLINE
    void
    operator()(const index_type& i, data_type& data) const
    {
      if (m_array[i] < data) {
        data = m_array[i];
      }
    }

    PUGS_INLINE
    void
    join(volatile data_type& dst, const volatile data_type& src) const
    {
      if (src < dst) {
        dst = src;
      }
    }

    PUGS_INLINE
    void
    init(data_type& value) const
    {
      value = std::numeric_limits<data_type>::max();
    }

    PUGS_INLINE
    ArrayMin(const ArrayType& array) : m_array(array)
    {
      ;
    }

    PUGS_INLINE
    ~ArrayMin() = default;
  };

  return ArrayMin(array);
}

template <typename DataType, template <typename> typename ArrayT>
std::remove_const_t<DataType>
max(const ArrayT<DataType>& array)
{
  using ArrayType  = ArrayT<DataType>;
  using data_type  = std::remove_const_t<typename ArrayType::data_type>;
  using index_type = typename ArrayType::index_type;

  class ArrayMax
  {
   private:
    const ArrayType m_array;

   public:
    PUGS_INLINE
    operator data_type()
    {
      data_type reduced_value;
      parallel_reduce(m_array.size(), *this, reduced_value);
      return reduced_value;
    }

    PUGS_INLINE
    void
    operator()(const index_type& i, data_type& data) const
    {
      if (m_array[i] > data) {
        data = m_array[i];
      }
    }

    PUGS_INLINE
    void
    join(volatile data_type& dst, const volatile data_type& src) const
    {
      if (src > dst) {
        dst = src;
      }
    }

    PUGS_INLINE
    void
    init(data_type& value) const
    {
      value = std::numeric_limits<data_type>::min();
    }

    PUGS_INLINE
    ArrayMax(const ArrayType& array) : m_array(array)
    {
      ;
    }

    PUGS_INLINE
    ~ArrayMax() = default;
  };

  return ArrayMax(array);
}

template <typename DataType, template <typename> typename ArrayT>
std::remove_const_t<DataType>
sum(const ArrayT<DataType>& array)
{
  using ArrayType  = ArrayT<DataType>;
  using data_type  = std::remove_const_t<DataType>;
  using index_type = typename ArrayType::index_type;

  class ArraySum
  {
   private:
    const ArrayType& m_array;

   public:
    PUGS_INLINE
    operator data_type()
    {
      data_type reduced_value;
      parallel_reduce(m_array.size(), *this, reduced_value);
      return reduced_value;
    }

    PUGS_INLINE
    void
    operator()(const index_type& i, data_type& data) const
    {
      data += m_array[i];
    }

    PUGS_INLINE
    void
    join(volatile data_type& dst, const volatile data_type& src) const
    {
      dst += src;
    }

    PUGS_INLINE
    void
    init(data_type& value) const
    {
      if constexpr (std::is_arithmetic_v<data_type>) {
        value = 0;
      } else {
        value = zero;
      }
    }

    PUGS_INLINE
    ArraySum(const ArrayType& array) : m_array(array)
    {
      ;
    }

    PUGS_INLINE
    ~ArraySum() = default;
  };

  return ArraySum(array);
}

template <template <typename... SourceT> typename SourceArray,
          template <typename... ImageT>
          typename ImageArray,
          typename... SourceT,
          typename... ImageT>
void
value_copy(const SourceArray<SourceT...>& source_array,
           ImageArray<ImageT...>& image_array)
{
  using SourceDataType = typename SourceArray<SourceT...>::data_type;
  using ImageDataType  = typename ImageArray<ImageT...>::data_type;

  static_assert(
    std::is_same_v<std::remove_const_t<SourceDataType>, ImageDataType>);
  static_assert(not std::is_const_v<ImageDataType>);

  Assert(source_array.size() == image_array.size());

  parallel_for(source_array.size(), PUGS_LAMBDA(const size_t& i) {
    image_array[i] = source_array[i];
  });
}

#endif   // ARRAY_UTILS_HPP
