#include <Connectivity.hpp>

#include <ConnectivityComputer.hpp>
#include <map>

template <typename ConnectivityType>
PUGS_INLINE ConnectivityMatrix
ConnectivityComputer::computeConnectivityMatrix(
  const ConnectivityType& connectivity,
  const ItemType& item_type,
  const ItemType& child_item_type) const
{
  ConnectivityMatrix item_to_child_item_matrix;
  if (connectivity.isConnectivityMatrixBuilt(child_item_type, item_type)) {
    const ConnectivityMatrix& child_to_item_matrix =
      connectivity._getMatrix(child_item_type, item_type);

    pout() << "computing connectivity " << itemName(item_type) << " -> "
           << itemName(child_item_type) << '\n';

    item_to_child_item_matrix = this->_computeInverse(child_to_item_matrix);
  } else {
    perr() << "unable to compute connectivity " << itemName(item_type) << " -> "
           << itemName(child_item_type) << '\n';
    std::terminate();
  }

  return item_to_child_item_matrix;
}

template ConnectivityMatrix ConnectivityComputer::computeConnectivityMatrix(
  const Connectivity1D&,
  const ItemType&,
  const ItemType&) const;

template ConnectivityMatrix ConnectivityComputer::computeConnectivityMatrix(
  const Connectivity2D&,
  const ItemType&,
  const ItemType&) const;

template ConnectivityMatrix ConnectivityComputer::computeConnectivityMatrix(
  const Connectivity3D&,
  const ItemType&,
  const ItemType&) const;

ConnectivityMatrix
ConnectivityComputer::_computeInverse(
  const ConnectivityMatrix& item_to_child_matrix) const
{
  std::map<unsigned int, std::vector<unsigned int>> child_to_item_vector_map;
  const size_t& number_of_items = item_to_child_matrix.numRows();

  for (unsigned int j = 0; j < number_of_items; ++j) {
    const auto& item_to_child = item_to_child_matrix.rowConst(j);
    for (unsigned int r = 0; r < item_to_child.length; ++r) {
      child_to_item_vector_map[item_to_child(r)].push_back(j);
    }
  }

  {
    size_t i = 0;
    for (const auto& [child_item_id, item_vector] : child_to_item_vector_map) {
      if (child_item_id != i) {
        perr() << "sparse item numerotation NIY\n";
        std::exit(0);
      }
      ++i;
    }
  }

  std::vector<std::vector<unsigned int>> child_to_items_vector(
    child_to_item_vector_map.size());
  for (const auto& [child_item_id, item_vector] : child_to_item_vector_map) {
    child_to_items_vector[child_item_id] = item_vector;
  }

  return ConnectivityMatrix(child_to_items_vector);
}

template <typename ItemOfItem, typename ConnectivityType>
WeakSubItemValuePerItem<const unsigned short, typename ItemOfItem::Reversed>
ConnectivityComputer::computeLocalItemNumberInChildItem(
  const ConnectivityType& connectivity) const
{
  using ReversedItemOfItem = typename ItemOfItem::Reversed;

  constexpr ItemType item_type       = ReversedItemOfItem::item_type;
  constexpr ItemType child_item_type = ReversedItemOfItem::sub_item_type;

  const ConnectivityMatrix& child_item_to_items_matrix =
    connectivity._getMatrix(child_item_type, item_type);
  const ConnectivityMatrix& item_to_child_items_matrix =
    connectivity._getMatrix(item_type, child_item_type);

  WeakSubItemValuePerItem<unsigned short, ReversedItemOfItem>
    item_number_in_child_item(connectivity);
  for (ItemIdT<item_type> r = 0; r < item_to_child_items_matrix.numRows();
       ++r) {
    const auto& item_to_child_items = item_to_child_items_matrix.rowConst(r);
    for (unsigned short J = 0; J < item_to_child_items.length; ++J) {
      const unsigned int j            = item_to_child_items(J);
      const auto& child_item_to_items = child_item_to_items_matrix.rowConst(j);

      for (unsigned int R = 0; R < child_item_to_items.length; ++R) {
        if (child_item_to_items(R) == r) {
          item_number_in_child_item(r, J) = R;
          break;
        }
      }
    }
  }

  return item_number_in_child_item;
}

// 1D

template WeakSubItemValuePerItem<const unsigned short, CellOfNode>
ConnectivityComputer::computeLocalItemNumberInChildItem<NodeOfCell>(
  const Connectivity1D&) const;

template WeakSubItemValuePerItem<const unsigned short, NodeOfCell>
ConnectivityComputer::computeLocalItemNumberInChildItem<CellOfNode>(
  const Connectivity1D&) const;

// 2D

template WeakSubItemValuePerItem<const unsigned short, CellOfNode>
ConnectivityComputer::computeLocalItemNumberInChildItem<NodeOfCell>(
  const Connectivity2D&) const;

template WeakSubItemValuePerItem<const unsigned short, CellOfFace>
ConnectivityComputer::computeLocalItemNumberInChildItem<FaceOfCell>(
  const Connectivity2D&) const;

template WeakSubItemValuePerItem<const unsigned short, FaceOfNode>
ConnectivityComputer::computeLocalItemNumberInChildItem<NodeOfFace>(
  const Connectivity2D&) const;

template WeakSubItemValuePerItem<const unsigned short, FaceOfCell>
ConnectivityComputer::computeLocalItemNumberInChildItem<CellOfFace>(
  const Connectivity2D&) const;

template WeakSubItemValuePerItem<const unsigned short, NodeOfFace>
ConnectivityComputer::computeLocalItemNumberInChildItem<FaceOfNode>(
  const Connectivity2D&) const;

template WeakSubItemValuePerItem<const unsigned short, NodeOfCell>
ConnectivityComputer::computeLocalItemNumberInChildItem<CellOfNode>(
  const Connectivity2D&) const;

// 3D

template WeakSubItemValuePerItem<const unsigned short, CellOfNode>
ConnectivityComputer::computeLocalItemNumberInChildItem<NodeOfCell>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, CellOfEdge>
ConnectivityComputer::computeLocalItemNumberInChildItem<EdgeOfCell>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, CellOfFace>
ConnectivityComputer::computeLocalItemNumberInChildItem<FaceOfCell>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, FaceOfNode>
ConnectivityComputer::computeLocalItemNumberInChildItem<NodeOfFace>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, FaceOfEdge>
ConnectivityComputer::computeLocalItemNumberInChildItem<EdgeOfFace>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, FaceOfCell>
ConnectivityComputer::computeLocalItemNumberInChildItem<CellOfFace>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, EdgeOfNode>
ConnectivityComputer::computeLocalItemNumberInChildItem<NodeOfEdge>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, EdgeOfFace>
ConnectivityComputer::computeLocalItemNumberInChildItem<FaceOfEdge>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, EdgeOfCell>
ConnectivityComputer::computeLocalItemNumberInChildItem<CellOfEdge>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, NodeOfEdge>
ConnectivityComputer::computeLocalItemNumberInChildItem<EdgeOfNode>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, NodeOfFace>
ConnectivityComputer::computeLocalItemNumberInChildItem<FaceOfNode>(
  const Connectivity3D&) const;

template WeakSubItemValuePerItem<const unsigned short, NodeOfCell>
ConnectivityComputer::computeLocalItemNumberInChildItem<CellOfNode>(
  const Connectivity3D&) const;
