#ifndef CONNECTIVITY_DESCRIPTOR_HPP
#define CONNECTIVITY_DESCRIPTOR_HPP

#include <ItemOfItemType.hpp>
#include <PugsTraits.hpp>
#include <RefItemList.hpp>

#include <vector>

class ConnectivityDescriptor
{
 private:
  std::vector<RefCellList> m_ref_cell_list_vector;
  std::vector<RefFaceList> m_ref_face_list_vector;
  std::vector<RefEdgeList> m_ref_edge_list_vector;
  std::vector<RefNodeList> m_ref_node_list_vector;

 public:
  std::vector<std::vector<unsigned int>> cell_to_node_vector;
  std::vector<std::vector<unsigned int>> cell_to_face_vector;
  std::vector<std::vector<unsigned int>> cell_to_edge_vector;

  std::vector<std::vector<unsigned int>> face_to_node_vector;
  std::vector<std::vector<unsigned int>> face_to_edge_vector;

  std::vector<std::vector<unsigned int>> edge_to_node_vector;

  template <typename ItemOfItemT>
  auto&
  itemOfItemVector()
  {
    if constexpr (std::is_same_v<ItemOfItemT, NodeOfCell>) {
      return cell_to_node_vector;
    } else if constexpr (std::is_same_v<ItemOfItemT, FaceOfCell>) {
      return cell_to_face_vector;
    } else if constexpr (std::is_same_v<ItemOfItemT, EdgeOfCell>) {
      return cell_to_edge_vector;
    } else if constexpr (std::is_same_v<ItemOfItemT, EdgeOfFace>) {
      return face_to_edge_vector;
    } else if constexpr (std::is_same_v<ItemOfItemT, NodeOfFace>) {
      return face_to_node_vector;
    } else if constexpr (std::is_same_v<ItemOfItemT, NodeOfEdge>) {
      return edge_to_node_vector;
    } else {
      static_assert(is_false_v<ItemOfItemT>, "Unexpected item of item type");
    }
  }

  std::vector<Array<bool>> cell_face_is_reversed_vector;
  std::vector<Array<bool>> face_edge_is_reversed_vector;

  std::vector<CellType> cell_type_vector;

  std::vector<int> cell_number_vector;
  std::vector<int> face_number_vector;
  std::vector<int> edge_number_vector;
  std::vector<int> node_number_vector;

  template <ItemType item_type>
  const std::vector<int>&
  itemNumberVector() const
  {
    if constexpr (item_type == ItemType::cell) {
      return cell_number_vector;
    } else if constexpr (item_type == ItemType::face) {
      return face_number_vector;
    } else if constexpr (item_type == ItemType::edge) {
      return edge_number_vector;
    } else if constexpr (item_type == ItemType::node) {
      return node_number_vector;
    } else {
      static_assert(is_false_item_type_v<item_type>, "Unexpected item type");
    }
  }

  std::vector<int> cell_owner_vector;
  std::vector<int> face_owner_vector;
  std::vector<int> edge_owner_vector;
  std::vector<int> node_owner_vector;

  template <ItemType item_type>
  const std::vector<RefItemList<item_type>>&
  refItemListVector() const
  {
    if constexpr (item_type == ItemType::cell) {
      return m_ref_cell_list_vector;
    } else if constexpr (item_type == ItemType::face) {
      return m_ref_face_list_vector;
    } else if constexpr (item_type == ItemType::edge) {
      return m_ref_edge_list_vector;
    } else if constexpr (item_type == ItemType::node) {
      return m_ref_node_list_vector;
    } else {
      static_assert(is_false_item_type_v<item_type>, "Unexpected item type");
    }
  }

  template <ItemType item_type>
  void
  addRefItemList(const RefItemList<item_type>& ref_item_list)
  {
    if constexpr (item_type == ItemType::cell) {
      m_ref_cell_list_vector.push_back(ref_item_list);
    } else if constexpr (item_type == ItemType::face) {
      m_ref_face_list_vector.push_back(ref_item_list);
    } else if constexpr (item_type == ItemType::edge) {
      m_ref_edge_list_vector.push_back(ref_item_list);
    } else if constexpr (item_type == ItemType::node) {
      m_ref_node_list_vector.push_back(ref_item_list);
    } else {
      static_assert(is_false_item_type_v<item_type>, "Unexpected item type");
    }
  }

  ConnectivityDescriptor& operator=(const ConnectivityDescriptor&) = delete;
  ConnectivityDescriptor& operator=(ConnectivityDescriptor&&) = delete;

  ConnectivityDescriptor()                              = default;
  ConnectivityDescriptor(const ConnectivityDescriptor&) = default;
  ConnectivityDescriptor(ConnectivityDescriptor&&)      = delete;
  ~ConnectivityDescriptor()                             = default;
};

#endif   // CONNECTIVITY_DESCRIPTOR_HPP
