#include <SignalManager.hpp>

#include <PastisAssert.hpp>
#include <PastisOStream.hpp>

#include <BacktraceManager.hpp>
#include <ConsoleManager.hpp>

#include <csignal>
#include <unistd.h>

#include <pastis_config.hpp>
#include <rang.hpp>

std::string SignalManager::s_pause_on_error = "auto";
void SignalManager::setPauseForDebug(const std::string& pause_on_error)
{
  s_pause_on_error = pause_on_error;
}

std::string SignalManager::signalName(const int& signal)
{
  switch (signal) {
  case SIGILL:  return "SIGILL";
  case SIGFPE:  return "SIGFPE";
  case SIGABRT: return "SIGABRT";
  case SIGINT:  return "SIGINT";
  case SIGSEGV: return "SIGSEGV";
  case SIGTERM: return "SIGTERM";
  }
  return "SIGNAL undefined!";
}

void SignalManager::pauseForDebug(const int& signal)
{
  if (std::string(PASTIS_BUILD_TYPE) != "Release") {
    if ((ConsoleManager::isTerminal(pout()) and
         (s_pause_on_error == "auto")) or
        (s_pause_on_error == "yes")) {
      pout() << "\n======================================\n";
      pout() << rang::style::reset
             << rang::fg::reset
             << rang::style::bold;
      pout() << "to attach gdb to this process run\n";
      pout() << "\tgdb -pid "
             << rang::fg::red
             << getpid()
             << rang::fg::reset
             << '\n';
      pout() << "else press Control-C to exit\n";
      pout() << rang::style::reset;
      pout() << "======================================\n";
      pout() << std::flush;

      pause();
    }
  }
  std::exit(signal);
}

void SignalManager::handler(int signal)
{
  std::signal(SIGFPE,  SIG_DFL);
  std::signal(SIGSEGV, SIG_IGN);
  std::signal(SIGTERM, SIG_DFL);
  std::signal(SIGINT,  SIG_DFL);
  std::signal(SIGABRT, SIG_DFL);

  std::exception_ptr eptr = std::current_exception();
  try {
    if (eptr) {
      std::rethrow_exception(eptr);
    }
  }
  catch(const AssertError& assert_error) {
    perr() << assert_error << '\n';
  }
  catch(...) {
    perr() << "Unknown exception!\n";
  }

  perr() << "\n *** "
	    << rang::style::reset
	    << rang::fg::reset
	    << rang::style::bold;
  perr() << "Signal "
    	    << rang::fgB::red
	    << signalName(signal)
    	    << rang::fg::reset
	    << " caught"
	    << rang::style::reset
	    << " ***\n";

  BacktraceManager bm;
  perr() << bm << '\n';

  SignalManager::pauseForDebug(signal);
 }

void SignalManager::init(const bool& enable)
{

  if (enable) {
    std::signal(SIGFPE,  SignalManager::handler);
    std::signal(SIGSEGV, SignalManager::handler);
    std::signal(SIGTERM, SignalManager::handler);
    std::signal(SIGINT,  SignalManager::handler);
    std::signal(SIGABRT, SignalManager::handler);
    std::signal(SIGPIPE, SignalManager::handler);

    pout() << "Signal management: "
	      << rang::style::bold
	      << rang::fgB::green
	      << "enabled"
	      << rang::fg::reset
	      << rang::style::reset << '\n';
  } else {
    pout() << "Signal management: "
	      << rang::style::bold
	      << rang::fgB::red
	      << "disabled"
	      << rang::fg::reset
	      << rang::style::reset << '\n';
  }
}
