#ifndef OUTPUT_NAMED_ITEM_DATA_SET_HPP
#define OUTPUT_NAMED_ITEM_DATA_SET_HPP

#include <mesh/ItemArray.hpp>
#include <mesh/ItemValue.hpp>

#include <algebra/TinyMatrix.hpp>
#include <algebra/TinyVector.hpp>

#include <map>
#include <string>
#include <variant>

template <typename DataType,
          ItemType item_type,
          typename ConnectivityPtr,
          template <typename DataTypeT, ItemType item_type_t, typename ConnectivityPtrT>
          typename ItemDataT>
class NamedItemData
{
 private:
  std::string m_name;
  using ItemDataType = ItemDataT<const DataType, item_type, std::shared_ptr<const IConnectivity>>;
  ItemDataType m_item_data;

 public:
  constexpr const std::string&
  name() const
  {
    return m_name;
  }

  constexpr const ItemDataType&
  itemData() const
  {
    return m_item_data;
  }

  NamedItemData& operator=(const NamedItemData&) = default;
  NamedItemData& operator=(NamedItemData&&) = default;

  NamedItemData(const std::string& name, const ItemDataT<DataType, item_type, ConnectivityPtr>& item_data)
    : m_name(name), m_item_data(item_data)
  {
    ;
  }

  NamedItemData(const std::string& name, const ItemDataT<const DataType, item_type, ConnectivityPtr>& item_data)
    : m_name(name), m_item_data(item_data)
  {
    ;
  }

  NamedItemData(const NamedItemData&) = default;
  NamedItemData(NamedItemData&&)      = default;
  ~NamedItemData()                    = default;
};

class OutputNamedItemDataSet
{
 public:
  using ItemDataVariant = std::variant<NodeValue<const bool>,
                                       NodeValue<const int>,
                                       NodeValue<const long int>,
                                       NodeValue<const unsigned long int>,
                                       NodeValue<const double>,
                                       NodeValue<const TinyVector<1, double>>,
                                       NodeValue<const TinyVector<2, double>>,
                                       NodeValue<const TinyVector<3, double>>,
                                       NodeValue<const TinyMatrix<1, double>>,
                                       NodeValue<const TinyMatrix<2, double>>,
                                       NodeValue<const TinyMatrix<3, double>>,

                                       CellValue<const bool>,
                                       CellValue<const int>,
                                       CellValue<const long int>,
                                       CellValue<const unsigned long int>,
                                       CellValue<const double>,
                                       CellValue<const TinyVector<1, double>>,
                                       CellValue<const TinyVector<2, double>>,
                                       CellValue<const TinyVector<3, double>>,
                                       CellValue<const TinyMatrix<1, double>>,
                                       CellValue<const TinyMatrix<2, double>>,
                                       CellValue<const TinyMatrix<3, double>>,

                                       NodeArray<const bool>,
                                       NodeArray<const int>,
                                       NodeArray<const long int>,
                                       NodeArray<const unsigned long int>,
                                       NodeArray<const double>,

                                       CellArray<const bool>,
                                       CellArray<const int>,
                                       CellArray<const long int>,
                                       CellArray<const unsigned long int>,
                                       CellArray<const double>>;

 private:
  std::map<std::string, ItemDataVariant> m_name_itemvariant_map;

  template <typename DataType,
            ItemType item_type,
            typename ConnectivityPtr,
            template <typename DataTypeT, ItemType item_type_t, typename ConnectivityPtrT>
            typename ItemDataT,
            typename... Args>
  PUGS_FORCEINLINE constexpr void
  _doInsert(const NamedItemData<DataType, item_type, ConnectivityPtr, ItemDataT>& named_item_data)
  {
    if (m_name_itemvariant_map.find(named_item_data.name()) == m_name_itemvariant_map.end()) {
      m_name_itemvariant_map[named_item_data.name()] = named_item_data.itemData();
    }
  }

  template <typename DataType,
            ItemType item_type,
            typename ConnectivityPtr,
            template <typename DataTypeT, ItemType item_type_t, typename ConnectivityPtrT>
            typename ItemDataT,
            typename... Args>
  PUGS_FORCEINLINE constexpr void
  _unpackVariadicInput(const NamedItemData<DataType, item_type, ConnectivityPtr, ItemDataT>& named_item_data,
                       Args&&... args)
  {
    _doInsert(named_item_data);
    if constexpr (sizeof...(args) > 0) {
      this->_unpackVariadicInput(std::forward<Args>(args)...);
    }
  }

 public:
  auto
  begin() const
  {
    return m_name_itemvariant_map.begin();
  }

  auto
  end() const
  {
    return m_name_itemvariant_map.end();
  }

  template <typename DataType,
            ItemType item_type,
            typename ConnectivityPtr,
            template <typename DataTypeT, ItemType item_type_t, typename ConnectivityPtrT>
            typename ItemDataT>
  void
  add(const NamedItemData<DataType, item_type, ConnectivityPtr, ItemDataT>& named_item_data)
  {
    _doInsert(named_item_data);
  }

  template <typename... DataType,
            ItemType... item_type,
            typename... ConnectivityPtr,
            template <typename DataTypeT, ItemType item_type_t, typename ConnectivityPtrT>
            typename... ItemDataT>
  OutputNamedItemDataSet(NamedItemData<DataType, item_type, ConnectivityPtr, ItemDataT>... named_item_data)
  {
    _unpackVariadicInput(named_item_data...);
  }

  OutputNamedItemDataSet(const OutputNamedItemDataSet&) = default;
  OutputNamedItemDataSet()                              = default;
  ~OutputNamedItemDataSet()                             = default;
};

#endif   // OUTPUT_NAMED_ITEM_DATA_SET_HPP
