#include <mesh/Connectivity.hpp>

#include <mesh/ConnectivityDescriptor.hpp>
#include <utils/Messenger.hpp>

#include <map>

template <size_t Dimension>
Connectivity<Dimension>::Connectivity()
{}

template <size_t Dimension>
void
Connectivity<Dimension>::_buildFrom(const ConnectivityDescriptor& descriptor)
{
  Assert(descriptor.cell_to_node_vector.size() == descriptor.cell_type_vector.size());
  Assert(descriptor.cell_number_vector.size() == descriptor.cell_type_vector.size());
  if constexpr (Dimension > 1) {
    Assert(descriptor.cell_to_face_vector.size() == descriptor.cell_type_vector.size());
    Assert(descriptor.face_to_node_vector.size() == descriptor.face_number_vector.size());
    Assert(descriptor.face_owner_vector.size() == descriptor.face_number_vector.size());
  }

  auto& cell_to_node_matrix = m_item_to_item_matrix[itemTId(ItemType::cell)][itemTId(ItemType::node)];
  cell_to_node_matrix       = descriptor.cell_to_node_vector;

  {
    WeakCellValue<CellType> cell_type(*this);
    parallel_for(
      this->numberOfCells(), PUGS_LAMBDA(CellId j) { cell_type[j] = descriptor.cell_type_vector[j]; });
    m_cell_type = cell_type;
  }

  {
    WeakCellValue<int> cell_number(*this);
    cell_number   = convert_to_array(descriptor.cell_number_vector);
    m_cell_number = cell_number;
  }

  {
    WeakNodeValue<int> node_number(*this);
    node_number   = convert_to_array(descriptor.node_number_vector);
    m_node_number = node_number;
  }

  {
    WeakCellValue<int> cell_global_index(*this);
    int first_index = 0;
    parallel_for(
      this->numberOfCells(), PUGS_LAMBDA(CellId j) { cell_global_index[j] = first_index + j; });
    m_cell_global_index = cell_global_index;
  }

  {
    WeakCellValue<int> cell_owner(*this);
    cell_owner   = convert_to_array(descriptor.cell_owner_vector);
    m_cell_owner = cell_owner;
  }

  {
    const int rank = parallel::rank();
    WeakCellValue<bool> cell_is_owned(*this);
    parallel_for(
      this->numberOfCells(), PUGS_LAMBDA(CellId j) { cell_is_owned[j] = (m_cell_owner[j] == rank); });
    m_cell_is_owned = cell_is_owned;
  }

  {
    WeakNodeValue<int> node_owner(*this);
    node_owner   = convert_to_array(descriptor.node_owner_vector);
    m_node_owner = node_owner;
  }

  {
    const int rank = parallel::rank();
    WeakNodeValue<bool> node_is_owned(*this);
    parallel_for(
      this->numberOfNodes(), PUGS_LAMBDA(NodeId r) { node_is_owned[r] = (m_node_owner[r] == rank); });
    m_node_is_owned = node_is_owned;
  }

  m_ref_node_list_vector = descriptor.template refItemListVector<ItemType::node>();
  m_ref_cell_list_vector = descriptor.template refItemListVector<ItemType::cell>();

  if constexpr (Dimension == 1) {
    // faces are similar to nodes
    {
      WeakFaceValue<int> face_number(*this);
      face_number   = convert_to_array(descriptor.node_number_vector);
      m_face_number = face_number;
    }

    {
      WeakFaceValue<int> face_owner(*this);
      face_owner   = convert_to_array(descriptor.node_owner_vector);
      m_face_owner = face_owner;
    }

    {
      const int rank = parallel::rank();
      WeakFaceValue<bool> face_is_owned(*this);
      parallel_for(
        this->numberOfFaces(), PUGS_LAMBDA(FaceId l) { face_is_owned[l] = (m_face_owner[l] == rank); });
      m_face_is_owned = face_is_owned;
    }

    // edges are similar to nodes
    {
      WeakEdgeValue<int> edge_number(*this);
      edge_number   = convert_to_array(descriptor.node_number_vector);
      m_edge_number = edge_number;
    }

    {
      WeakEdgeValue<int> edge_owner(*this);
      edge_owner   = convert_to_array(descriptor.node_owner_vector);
      m_edge_owner = edge_owner;
    }

    {
      const int rank = parallel::rank();
      WeakEdgeValue<bool> edge_is_owned(*this);
      parallel_for(
        this->numberOfEdges(), PUGS_LAMBDA(EdgeId l) { edge_is_owned[l] = (m_edge_owner[l] == rank); });
      m_edge_is_owned = edge_is_owned;
    }
  } else {
    m_item_to_item_matrix[itemTId(ItemType::face)][itemTId(ItemType::node)] = descriptor.face_to_node_vector;

    m_item_to_item_matrix[itemTId(ItemType::cell)][itemTId(ItemType::face)] = descriptor.cell_to_face_vector;

    {
      FaceValuePerCell<bool> cell_face_is_reversed(*this);
      for (CellId j = 0; j < descriptor.cell_face_is_reversed_vector.size(); ++j) {
        const auto& face_cells_vector = descriptor.cell_face_is_reversed_vector[j];
        for (unsigned short lj = 0; lj < face_cells_vector.size(); ++lj) {
          cell_face_is_reversed(j, lj) = face_cells_vector[lj];
        }
      }
      m_cell_face_is_reversed = cell_face_is_reversed;
    }

    {
      WeakFaceValue<int> face_number(*this);
      face_number   = convert_to_array(descriptor.face_number_vector);
      m_face_number = face_number;
    }

    {
      WeakFaceValue<int> face_owner(*this);
      face_owner   = convert_to_array(descriptor.face_owner_vector);
      m_face_owner = face_owner;
    }

    {
      const int rank = parallel::rank();
      WeakFaceValue<bool> face_is_owned(*this);
      parallel_for(
        this->numberOfFaces(), PUGS_LAMBDA(FaceId l) { face_is_owned[l] = (m_face_owner[l] == rank); });
      m_face_is_owned = face_is_owned;
    }

    m_ref_face_list_vector = descriptor.template refItemListVector<ItemType::face>();

    if constexpr (Dimension == 2) {
      // edges are similar to faces
      {
        WeakEdgeValue<int> edge_number(*this);
        edge_number   = convert_to_array(descriptor.face_number_vector);
        m_edge_number = edge_number;
      }

      {
        WeakEdgeValue<int> edge_owner(*this);
        edge_owner   = convert_to_array(descriptor.face_owner_vector);
        m_edge_owner = edge_owner;
      }

      {
        const int rank = parallel::rank();
        WeakEdgeValue<bool> edge_is_owned(*this);
        parallel_for(
          this->numberOfEdges(), PUGS_LAMBDA(EdgeId l) { edge_is_owned[l] = (m_edge_owner[l] == rank); });
        m_edge_is_owned = edge_is_owned;
      }

    } else {
      m_item_to_item_matrix[itemTId(ItemType::edge)][itemTId(ItemType::node)] = descriptor.edge_to_node_vector;

      m_item_to_item_matrix[itemTId(ItemType::face)][itemTId(ItemType::edge)] = descriptor.face_to_edge_vector;

      m_item_to_item_matrix[itemTId(ItemType::cell)][itemTId(ItemType::edge)] = descriptor.cell_to_edge_vector;

      {
        EdgeValuePerFace<bool> face_edge_is_reversed(*this);
        for (FaceId l = 0; l < descriptor.face_edge_is_reversed_vector.size(); ++l) {
          const auto& edge_faces_vector = descriptor.face_edge_is_reversed_vector[l];
          for (unsigned short el = 0; el < edge_faces_vector.size(); ++el) {
            face_edge_is_reversed(l, el) = edge_faces_vector[el];
          }
        }
        m_face_edge_is_reversed = face_edge_is_reversed;
      }

      {
        WeakEdgeValue<int> edge_number(*this);
        edge_number   = convert_to_array(descriptor.edge_number_vector);
        m_edge_number = edge_number;
      }

      {
        WeakEdgeValue<int> edge_owner(*this);
        edge_owner   = convert_to_array(descriptor.edge_owner_vector);
        m_edge_owner = edge_owner;
      }

      {
        const int rank = parallel::rank();
        WeakEdgeValue<bool> edge_is_owned(*this);
        parallel_for(
          this->numberOfEdges(), PUGS_LAMBDA(EdgeId e) { edge_is_owned[e] = (m_edge_owner[e] == rank); });
        m_edge_is_owned = edge_is_owned;
      }

      m_ref_edge_list_vector = descriptor.template refItemListVector<ItemType::edge>();
    }
  }
}

template Connectivity<1>::Connectivity();
template Connectivity<2>::Connectivity();
template Connectivity<3>::Connectivity();

template void Connectivity<1>::_buildFrom(const ConnectivityDescriptor&);
template void Connectivity<2>::_buildFrom(const ConnectivityDescriptor&);
template void Connectivity<3>::_buildFrom(const ConnectivityDescriptor&);
