#include <scheme/DiscreteFunctionInterpoler.hpp>

#include <language/utils/InterpolateItemValue.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <utils/Exceptions.hpp>

template <size_t Dimension, typename DataType>
std::shared_ptr<IDiscreteFunction>
DiscreteFunctionInterpoler::_interpolate() const
{
  std::shared_ptr mesh    = std::dynamic_pointer_cast<const Mesh<Connectivity<Dimension>>>(m_mesh);
  using MeshDataType      = MeshData<Dimension>;
  MeshDataType& mesh_data = MeshDataManager::instance().getMeshData(*mesh);

  return std::make_shared<
    DiscreteFunctionP0<Dimension, DataType>>(mesh,
                                             InterpolateItemValue<DataType(TinyVector<Dimension>)>::
                                               template interpolate<ItemType::cell>(m_function_id, mesh_data.xj()));
}

template <size_t Dimension>
std::shared_ptr<IDiscreteFunction>
DiscreteFunctionInterpoler::_interpolate() const
{
  const auto& function_descriptor = m_function_id.descriptor();
  Assert(function_descriptor.domainMappingNode().children[1]->m_data_type == ASTNodeDataType::typename_t);

  const ASTNodeDataType& data_type = function_descriptor.domainMappingNode().children[1]->m_data_type.contentType();

  switch (data_type) {
  case ASTNodeDataType::bool_t: {
    return this->_interpolate<Dimension, bool>();
  }
  case ASTNodeDataType::unsigned_int_t: {
    return this->_interpolate<Dimension, uint64_t>();
  }
  case ASTNodeDataType::int_t: {
    return this->_interpolate<Dimension, int64_t>();
  }
  case ASTNodeDataType::double_t: {
    return this->_interpolate<Dimension, double>();
  }
  case ASTNodeDataType::vector_t: {
    switch (data_type.dimension()) {
    case 1: {
      return this->_interpolate<Dimension, TinyVector<1>>();
    }
    case 2: {
      return this->_interpolate<Dimension, TinyVector<2>>();
    }
    case 3: {
      return this->_interpolate<Dimension, TinyVector<3>>();
    }
    default: {
      std::ostringstream os;
      os << "invalid vector dimension " << rang::fgB::red << data_type.dimension() << rang::style::reset;

      throw UnexpectedError(os.str());
    }
    }
  }
  case ASTNodeDataType::matrix_t: {
    Assert(data_type.numberOfColumns() == data_type.numberOfRows(), "undefined matrix type");
    switch (data_type.numberOfColumns()) {
    case 1: {
      return this->_interpolate<Dimension, TinyMatrix<1>>();
    }
    case 2: {
      return this->_interpolate<Dimension, TinyMatrix<2>>();
    }
    case 3: {
      return this->_interpolate<Dimension, TinyMatrix<3>>();
    }
    default: {
      std::ostringstream os;
      os << "invalid vector dimension " << rang::fgB::red << data_type.dimension() << rang::style::reset;

      throw UnexpectedError(os.str());
    }
    }
  }
  default: {
    std::ostringstream os;
    os << "invalid interpolation value type: " << rang::fgB::red << dataTypeName(data_type) << rang::style::reset;

    throw UnexpectedError(os.str());
  }
  }
}

std::shared_ptr<IDiscreteFunction>
DiscreteFunctionInterpoler::interpolate() const
{
  std::shared_ptr<IDiscreteFunction> discrete_function;
  switch (m_mesh->dimension()) {
  case 1: {
    return this->_interpolate<1>();
  }
  case 2: {
    return this->_interpolate<2>();
  }
  case 3: {
    return this->_interpolate<3>();
  }
  default: {
    throw UnexpectedError("invalid dimension");
  }
  }
  return nullptr;
}
