#include <mesh/CartesianMeshBuilder.hpp>

#include <mesh/Connectivity.hpp>
#include <mesh/ConnectivityDescriptor.hpp>
#include <mesh/ConnectivityDispatcher.hpp>
#include <mesh/RefId.hpp>
#include <utils/Array.hpp>
#include <utils/Messenger.hpp>

template <size_t Dimension>
inline void
CartesianMeshBuilder::_buildBoundaryNodeList(const TinyVector<Dimension, uint64_t>&, ConnectivityDescriptor&)
{
  static_assert(Dimension <= 3, "unexpected dimension");
}

template <>
inline void
CartesianMeshBuilder::_buildBoundaryNodeList(
  const TinyVector<1, uint64_t>& cell_size,   // clazy:exclude=function-args-by-value
  ConnectivityDescriptor& descriptor)
{
  {   // xmin
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = 0;
    descriptor.addRefItemList(RefNodeList{RefId{0, "XMIN"}, boundary_nodes});
  }

  {   // xmax
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = cell_size[0];
    descriptor.addRefItemList(RefNodeList{RefId{1, "XMAX"}, boundary_nodes});
  }
}

template <>
void
CartesianMeshBuilder::_buildBoundaryNodeList(
  const TinyVector<2, uint64_t>& cell_size,   // clazy:exclude=function-args-by-value
  ConnectivityDescriptor& descriptor)
{
  const TinyVector<2, uint64_t> node_size{cell_size[0] + 1, cell_size[1] + 1};

  const auto node_number = [&](const TinyVector<2, uint64_t> node_logic_id) {
    return node_logic_id[0] * node_size[1] + node_logic_id[1];
  };

  {   // xminymin
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({0, 0});
    descriptor.addRefItemList(RefNodeList{RefId{10, "XMINYMIN"}, boundary_nodes});
  }

  {   // xmaxymin
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({cell_size[0], 0});
    descriptor.addRefItemList(RefNodeList{RefId{11, "XMAXYMIN"}, boundary_nodes});
  }

  {   // xmaxymax
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({cell_size[0], cell_size[1]});
    descriptor.addRefItemList(RefNodeList{RefId{12, "XMAXYMAX"}, boundary_nodes});
  }

  {   // xminymax
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({0, cell_size[1]});
    descriptor.addRefItemList(RefNodeList{RefId{13, "XMINYMAX"}, boundary_nodes});
  }
}

template <>
void
CartesianMeshBuilder::_buildBoundaryNodeList(const TinyVector<3, uint64_t>& cell_size,
                                             ConnectivityDescriptor& descriptor)
{
  const TinyVector<3, uint64_t> node_size{cell_size[0] + 1, cell_size[1] + 1, cell_size[2] + 1};

  const auto node_number = [&](const TinyVector<3, uint64_t>& node_logic_id) {
    return (node_logic_id[0] * node_size[1] + node_logic_id[1]) * node_size[2] + node_logic_id[2];
  };

  {   // xminyminzmin
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({0, 0, 0});
    descriptor.addRefItemList(RefNodeList{RefId{10, "XMINYMINZMIN"}, boundary_nodes});
  }

  {   // xmaxyminzmin
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({cell_size[0], 0, 0});
    descriptor.addRefItemList(RefNodeList{RefId{11, "XMAXYMINZMIN"}, boundary_nodes});
  }

  {   // xmaxymaxzmin
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({cell_size[0], cell_size[1], 0});
    descriptor.addRefItemList(RefNodeList{RefId{12, "XMAXYMAXZMIN"}, boundary_nodes});
  }

  {   // xminymaxzmin
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({0, cell_size[1], 0});
    descriptor.addRefItemList(RefNodeList{RefId{13, "XMINYMAXZMIN"}, boundary_nodes});
  }

  {   // xminyminzmax
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({0, 0, cell_size[2]});
    descriptor.addRefItemList(RefNodeList{RefId{14, "XMINYMINZMAX"}, boundary_nodes});
  }

  {   // xmaxyminzmax
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({cell_size[0], 0, cell_size[2]});
    descriptor.addRefItemList(RefNodeList{RefId{15, "XMAXYMINZMAX"}, boundary_nodes});
  }

  {   // xmaxymaxzmax
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({cell_size[0], cell_size[1], cell_size[2]});
    descriptor.addRefItemList(RefNodeList{RefId{16, "XMAXYMAXZMAX"}, boundary_nodes});
  }

  {   // xminymaxzmax
    Array<NodeId> boundary_nodes(1);
    boundary_nodes[0] = node_number({0, cell_size[1], cell_size[2]});
    descriptor.addRefItemList(RefNodeList{RefId{17, "XMINYMAXZMAX"}, boundary_nodes});
  }
}

template <size_t Dimension>
void
CartesianMeshBuilder::_buildBoundaryEdgeList(const TinyVector<Dimension, uint64_t>&, ConnectivityDescriptor&)
{
  static_assert(Dimension == 3, "unexpected dimension");
}

template <>
void
CartesianMeshBuilder::_buildBoundaryEdgeList(const TinyVector<3, uint64_t>& cell_size,
                                             ConnectivityDescriptor& descriptor)
{
  using Edge                                                                 = ConnectivityFace<2>;
  const auto& node_number_vector                                             = descriptor.node_number_vector;
  const std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map = [&] {
    std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map;
    for (EdgeId l = 0; l < descriptor.edge_to_node_vector.size(); ++l) {
      const auto& node_vector                               = descriptor.edge_to_node_vector[l];
      edge_to_id_map[Edge(node_vector, node_number_vector)] = l;
    }
    return edge_to_id_map;
  }();

  std::unordered_map<int, EdgeId> edge_number_id_map = [&] {
    std::unordered_map<int, EdgeId> edge_number_id_map;
    for (size_t l = 0; l < descriptor.edge_number_vector.size(); ++l) {
      edge_number_id_map[descriptor.edge_number_vector[l]] = l;
    }
    Assert(edge_number_id_map.size() == descriptor.edge_number_vector.size());
    return edge_number_id_map;
  }();

  const TinyVector<3, uint64_t> node_size{cell_size[0] + 1, cell_size[1] + 1, cell_size[2] + 1};
  const auto node_number = [&](const TinyVector<3, uint64_t>& node_logic_id) {
    return (node_logic_id[0] * node_size[1] + node_logic_id[1]) * node_size[2] + node_logic_id[2];
  };

  {   // Ridge edges in direction of Z axis
    const auto add_ref_item_list_along_z = [&](const size_t i, const size_t j, const unsigned ref_id,
                                               const std::string& ref_name) {
      Array<EdgeId> boundary_edges(cell_size[2]);
      size_t l = 0;
      for (size_t k = 0; k < cell_size[2]; ++k) {
        const uint32_t node_0_id = node_number(TinyVector<3, uint64_t>{i, j, k});
        const uint32_t node_1_id = node_number(TinyVector<3, uint64_t>{i, j, k + 1});

        auto i_edge = edge_to_id_map.find(Edge{{node_0_id, node_1_id}, descriptor.node_number_vector});
        Assert(i_edge != edge_to_id_map.end());

        boundary_edges[l++] = i_edge->second;
      }
      Assert(l == cell_size[2]);
      descriptor.addRefItemList(RefEdgeList{RefId{ref_id, ref_name}, boundary_edges});
    };

    add_ref_item_list_along_z(0, 0, 20, "XMINYMIN");
    add_ref_item_list_along_z(0, cell_size[1], 21, "XMINYMAX");
    add_ref_item_list_along_z(cell_size[0], cell_size[1], 22, "XMAXYMAX");
    add_ref_item_list_along_z(cell_size[0], 0, 23, "XMAXYMIN");
  }

  {   // Ridge edges in direction of Y axis
    const auto add_ref_item_list_along_y = [&](const size_t i, const size_t k, const unsigned ref_id,
                                               const std::string& ref_name) {
      Array<EdgeId> boundary_edges(cell_size[1]);
      size_t l = 0;
      for (size_t j = 0; j < cell_size[1]; ++j) {
        const uint32_t node_0_id = node_number(TinyVector<3, uint64_t>{i, j, k});
        const uint32_t node_1_id = node_number(TinyVector<3, uint64_t>{i, j + 1, k});

        auto i_edge = edge_to_id_map.find(Edge{{node_0_id, node_1_id}, descriptor.node_number_vector});
        Assert(i_edge != edge_to_id_map.end());

        boundary_edges[l++] = i_edge->second;
      }
      Assert(l == cell_size[1]);
      descriptor.addRefItemList(RefEdgeList{RefId{ref_id, ref_name}, boundary_edges});
    };

    add_ref_item_list_along_y(0, 0, 24, "XMINZMIN");
    add_ref_item_list_along_y(0, cell_size[2], 25, "XMINZMAX");
    add_ref_item_list_along_y(cell_size[0], cell_size[2], 26, "XMAXZMAX");
    add_ref_item_list_along_y(cell_size[0], 0, 27, "XMAXZMIN");
  }

  {   // Ridge edges in direction of X axis
    const auto add_ref_item_list_along_x = [&](const size_t j, const size_t k, const unsigned ref_id,
                                               const std::string& ref_name) {
      Array<EdgeId> boundary_edges(cell_size[0]);
      size_t l = 0;
      for (size_t i = 0; i < cell_size[0]; ++i) {
        const uint32_t node_0_id = node_number(TinyVector<3, uint64_t>{i, j, k});
        const uint32_t node_1_id = node_number(TinyVector<3, uint64_t>{i + 1, j, k});

        auto i_edge = edge_to_id_map.find(Edge{{node_0_id, node_1_id}, descriptor.node_number_vector});
        Assert(i_edge != edge_to_id_map.end());

        boundary_edges[l++] = i_edge->second;
      }
      Assert(l == cell_size[0]);
      descriptor.addRefItemList(RefEdgeList{RefId{ref_id, ref_name}, boundary_edges});
    };

    add_ref_item_list_along_x(0, 0, 28, "YMINZMIN");
    add_ref_item_list_along_x(0, cell_size[2], 29, "YMINZMAX");
    add_ref_item_list_along_x(cell_size[1], cell_size[2], 30, "YMAXZMAX");
    add_ref_item_list_along_x(cell_size[1], 0, 31, "YMAXZMIN");
  }
}

template <size_t Dimension>
void
CartesianMeshBuilder::_buildBoundaryFaceList(const TinyVector<Dimension, uint64_t>&, ConnectivityDescriptor&)
{
  static_assert(Dimension >= 2 and Dimension <= 3, "unexpected dimension");
}

template <>
void
CartesianMeshBuilder::_buildBoundaryFaceList(
  const TinyVector<2, uint64_t>& cell_size,   // clazy:exclude=function-args-by-value
  ConnectivityDescriptor& descriptor)
{
  const auto cell_number = [&](const TinyVector<2, uint64_t> cell_logic_id) {
    return cell_logic_id[0] * cell_size[1] + cell_logic_id[1];
  };

  {   // xmin
    const size_t i = 0;
    Array<FaceId> boundary_faces(cell_size[1]);
    for (size_t j = 0; j < cell_size[1]; ++j) {
      constexpr size_t left_face = 3;

      const size_t cell_id = cell_number(TinyVector<2, uint64_t>{i, j});
      const size_t face_id = descriptor.cell_to_face_vector[cell_id][left_face];

      boundary_faces[j] = face_id;
    }
    descriptor.addRefItemList(RefFaceList{RefId{0, "XMIN"}, boundary_faces});
  }

  {   // xmax
    const size_t i = cell_size[0] - 1;
    Array<FaceId> boundary_faces(cell_size[1]);
    for (size_t j = 0; j < cell_size[1]; ++j) {
      constexpr size_t right_face = 1;

      const size_t cell_id = cell_number(TinyVector<2, uint64_t>{i, j});
      const size_t face_id = descriptor.cell_to_face_vector[cell_id][right_face];

      boundary_faces[j] = face_id;
    }
    descriptor.addRefItemList(RefFaceList{RefId{1, "XMAX"}, boundary_faces});
  }

  {   // ymin
    const size_t j = 0;
    Array<FaceId> boundary_faces(cell_size[0]);
    for (size_t i = 0; i < cell_size[0]; ++i) {
      constexpr size_t bottom_face = 0;

      const size_t cell_id = cell_number(TinyVector<2, uint64_t>{i, j});
      const size_t face_id = descriptor.cell_to_face_vector[cell_id][bottom_face];

      boundary_faces[i] = face_id;
    }
    descriptor.addRefItemList(RefFaceList{RefId{2, "YMIN"}, boundary_faces});
  }

  {   // ymax
    const size_t j = cell_size[1] - 1;
    Array<FaceId> boundary_faces(cell_size[0]);
    for (size_t i = 0; i < cell_size[0]; ++i) {
      constexpr size_t top_face = 2;

      const size_t cell_id = cell_number(TinyVector<2, uint64_t>{i, j});
      const size_t face_id = descriptor.cell_to_face_vector[cell_id][top_face];

      boundary_faces[i] = face_id;
    }
    descriptor.addRefItemList(RefFaceList{RefId{3, "YMAX"}, boundary_faces});
  }
}

template <>
void
CartesianMeshBuilder::_buildBoundaryFaceList(const TinyVector<3, uint64_t>& cell_size,
                                             ConnectivityDescriptor& descriptor)
{
  using Face = ConnectivityFace<3>;

  const std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map = [&] {
    std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map;
    for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
      const auto& node_vector                                          = descriptor.face_to_node_vector[l];
      face_to_id_map[Face(node_vector, descriptor.node_number_vector)] = l;
    }
    return face_to_id_map;
  }();

  const std::unordered_map<int, FaceId> face_number_id_map = [&] {
    std::unordered_map<int, FaceId> face_number_id_map;
    for (size_t l = 0; l < descriptor.face_number_vector.size(); ++l) {
      face_number_id_map[descriptor.face_number_vector[l]] = l;
    }
    Assert(face_number_id_map.size() == descriptor.face_number_vector.size());
    return face_number_id_map;
  }();

  const TinyVector<3, uint64_t> node_size{cell_size[0] + 1, cell_size[1] + 1, cell_size[2] + 1};
  const auto node_number = [&](const TinyVector<3, uint64_t>& node_logic_id) {
    return (node_logic_id[0] * node_size[1] + node_logic_id[1]) * node_size[2] + node_logic_id[2];
  };

  {   // faces orthogonal to X
    const auto add_ref_item_list_for_x = [&](const size_t i, const unsigned ref_id, const std::string& ref_name) {
      Array<FaceId> boundary_faces(cell_size[1] * cell_size[2]);
      size_t l = 0;
      for (size_t j = 0; j < cell_size[1]; ++j) {
        for (size_t k = 0; k < cell_size[2]; ++k) {
          const uint32_t node_0_id = node_number(TinyVector<3, uint64_t>{i, j, k});
          const uint32_t node_1_id = node_number(TinyVector<3, uint64_t>{i, j + 1, k});
          const uint32_t node_2_id = node_number(TinyVector<3, uint64_t>{i, j + 1, k + 1});
          const uint32_t node_3_id = node_number(TinyVector<3, uint64_t>{i, j, k + 1});

          auto i_face =
            face_to_id_map.find(Face{{node_0_id, node_1_id, node_2_id, node_3_id}, descriptor.node_number_vector});
          Assert(i_face != face_to_id_map.end());

          boundary_faces[l++] = i_face->second;
        }
      }
      Assert(l == cell_size[1] * cell_size[2]);
      descriptor.addRefItemList(RefFaceList{RefId{ref_id, ref_name}, boundary_faces});
    };

    add_ref_item_list_for_x(0, 0, "XMIN");
    add_ref_item_list_for_x(cell_size[0], 1, "XMAX");
  }

  {   // faces orthogonal to Y
    const auto add_ref_item_list_for_y = [&](const size_t j, const unsigned ref_id, const std::string& ref_name) {
      Array<FaceId> boundary_faces(cell_size[0] * cell_size[2]);
      size_t l = 0;
      for (size_t i = 0; i < cell_size[0]; ++i) {
        for (size_t k = 0; k < cell_size[2]; ++k) {
          const uint32_t node_0_id = node_number(TinyVector<3, uint64_t>{i, j, k});
          const uint32_t node_1_id = node_number(TinyVector<3, uint64_t>{i + 1, j, k});
          const uint32_t node_2_id = node_number(TinyVector<3, uint64_t>{i + 1, j, k + 1});
          const uint32_t node_3_id = node_number(TinyVector<3, uint64_t>{i, j, k + 1});

          auto i_face =
            face_to_id_map.find(Face{{node_0_id, node_1_id, node_2_id, node_3_id}, descriptor.node_number_vector});
          Assert(i_face != face_to_id_map.end());

          boundary_faces[l++] = i_face->second;
        }
      }
      Assert(l == cell_size[0] * cell_size[2]);
      descriptor.addRefItemList(RefFaceList{RefId{ref_id, ref_name}, boundary_faces});
    };

    add_ref_item_list_for_y(0, 2, "YMIN");
    add_ref_item_list_for_y(cell_size[1], 3, "YMAX");
  }

  {   // faces orthogonal to Z
    const auto add_ref_item_list_for_z = [&](const size_t k, const unsigned ref_id, const std::string& ref_name) {
      Array<FaceId> boundary_faces(cell_size[0] * cell_size[1]);
      size_t l = 0;
      for (size_t i = 0; i < cell_size[0]; ++i) {
        for (size_t j = 0; j < cell_size[1]; ++j) {
          const uint32_t node_0_id = node_number(TinyVector<3, uint64_t>{i, j, k});
          const uint32_t node_1_id = node_number(TinyVector<3, uint64_t>{i + 1, j, k});
          const uint32_t node_2_id = node_number(TinyVector<3, uint64_t>{i + 1, j + 1, k});
          const uint32_t node_3_id = node_number(TinyVector<3, uint64_t>{i, j + 1, k});

          auto i_face =
            face_to_id_map.find(Face{{node_0_id, node_1_id, node_2_id, node_3_id}, descriptor.node_number_vector});
          Assert(i_face != face_to_id_map.end());

          boundary_faces[l++] = i_face->second;
        }
      }
      Assert(l == cell_size[0] * cell_size[1]);
      descriptor.addRefItemList(RefFaceList{RefId{ref_id, ref_name}, boundary_faces});
    };

    add_ref_item_list_for_z(0, 4, "ZMIN");
    add_ref_item_list_for_z(cell_size[2], 5, "ZMAX");
  }
}

template <>
void
CartesianMeshBuilder::_buildCartesianMesh(
  const TinyVector<1>& a,                     // clazy:exclude=function-args-by-value
  const TinyVector<1>& b,                     // clazy:exclude=function-args-by-value
  const TinyVector<1, uint64_t>& cell_size)   // clazy:exclude=function-args-by-value
{
  const size_t number_of_cells = cell_size[0];
  const size_t number_of_nodes = cell_size[0] + 1;

  ConnectivityDescriptor descriptor;
  descriptor.node_number_vector.resize(number_of_nodes);
  for (size_t i = 0; i < number_of_nodes; ++i) {
    descriptor.node_number_vector[i] = i;
  }

  descriptor.cell_number_vector.resize(number_of_cells);
  for (size_t i = 0; i < number_of_cells; ++i) {
    descriptor.cell_number_vector[i] = i;
  }

  descriptor.cell_type_vector.resize(number_of_cells);
  std::fill(descriptor.cell_type_vector.begin(), descriptor.cell_type_vector.end(), CellType::Line);

  descriptor.cell_to_node_vector.resize(number_of_cells);
  constexpr size_t nb_node_per_cell = 2;
  for (size_t j = 0; j < number_of_cells; ++j) {
    descriptor.cell_to_node_vector[j].resize(nb_node_per_cell);
    for (size_t r = 0; r < nb_node_per_cell; ++r) {
      descriptor.cell_to_node_vector[j][0] = j;
      descriptor.cell_to_node_vector[j][1] = j + 1;
    }
  }

  this->_buildBoundaryNodeList(cell_size, descriptor);

  descriptor.cell_owner_vector.resize(number_of_cells);
  std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

  descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
  std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

  std::shared_ptr p_connectivity = Connectivity1D::build(descriptor);
  Connectivity1D& connectivity   = *p_connectivity;

  const double h = (b[0] - a[0]) / cell_size[0];

  NodeValue<TinyVector<1>> xr(connectivity);
  for (NodeId r = 0; r < number_of_nodes; ++r) {
    xr[r] = a + r * h;
  }

  m_mesh = std::make_shared<Mesh<Connectivity1D>>(p_connectivity, xr);
}

template <>
void
CartesianMeshBuilder::_buildCartesianMesh(
  const TinyVector<2>& a,                     // clazy:exclude=function-args-by-value
  const TinyVector<2>& b,                     // clazy:exclude=function-args-by-value
  const TinyVector<2, uint64_t>& cell_size)   // clazy:exclude=function-args-by-value
{
  constexpr size_t Dimension = 2;

  const TinyVector<Dimension, uint64_t> node_size{cell_size[0] + 1, cell_size[1] + 1};

  const size_t number_of_cells = cell_size[0] * cell_size[1];
  const size_t number_of_nodes = node_size[0] * node_size[1];

  ConnectivityDescriptor descriptor;
  descriptor.node_number_vector.resize(number_of_nodes);
  for (size_t i = 0; i < number_of_nodes; ++i) {
    descriptor.node_number_vector[i] = i;
  }

  descriptor.cell_number_vector.resize(number_of_cells);
  for (size_t i = 0; i < number_of_cells; ++i) {
    descriptor.cell_number_vector[i] = i;
  }

  descriptor.cell_type_vector.resize(number_of_cells);
  std::fill(descriptor.cell_type_vector.begin(), descriptor.cell_type_vector.end(), CellType::Quadrangle);

  const auto node_logic_id = [&](size_t r) {
    const uint64_t r0 = r / node_size[1];
    const uint64_t r1 = r % node_size[1];
    return TinyVector<Dimension, uint64_t>{r0, r1};
  };

  const auto node_number = [&](const TinyVector<Dimension, uint64_t> node_logic_id) {
    return node_logic_id[0] * node_size[1] + node_logic_id[1];
  };

  const auto cell_logic_id = [&](size_t j) {
    const uint64_t j0 = j / cell_size[1];
    const uint64_t j1 = j % cell_size[1];
    return TinyVector<Dimension, uint64_t>{j0, j1};
  };

  descriptor.cell_to_node_vector.resize(number_of_cells);
  constexpr size_t nb_node_per_cell = 1 << Dimension;
  for (size_t j = 0; j < number_of_cells; ++j) {
    TinyVector<Dimension, size_t> cell_index = cell_logic_id(j);
    descriptor.cell_to_node_vector[j].resize(nb_node_per_cell);
    for (size_t r = 0; r < nb_node_per_cell; ++r) {
      descriptor.cell_to_node_vector[j][0] = node_number(cell_index + TinyVector<Dimension, uint64_t>{0, 0});
      descriptor.cell_to_node_vector[j][1] = node_number(cell_index + TinyVector<Dimension, uint64_t>{1, 0});
      descriptor.cell_to_node_vector[j][2] = node_number(cell_index + TinyVector<Dimension, uint64_t>{1, 1});
      descriptor.cell_to_node_vector[j][3] = node_number(cell_index + TinyVector<Dimension, uint64_t>{0, 1});
    }
  }

  MeshBuilderBase::_computeCellFaceAndFaceNodeConnectivities<Dimension>(descriptor);

  this->_buildBoundaryNodeList(cell_size, descriptor);
  this->_buildBoundaryFaceList(cell_size, descriptor);

  descriptor.cell_owner_vector.resize(number_of_cells);
  std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

  descriptor.face_owner_vector.resize(descriptor.face_number_vector.size());
  std::fill(descriptor.face_owner_vector.begin(), descriptor.face_owner_vector.end(), parallel::rank());

  descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
  std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

  std::shared_ptr p_connectivity        = Connectivity<Dimension>::build(descriptor);
  Connectivity<Dimension>& connectivity = *p_connectivity;

  const TinyVector<Dimension> h{(b[0] - a[0]) / cell_size[0], (b[1] - a[1]) / cell_size[1]};

  NodeValue<TinyVector<Dimension>> xr(connectivity);
  for (NodeId r = 0; r < number_of_nodes; ++r) {
    const TinyVector<Dimension, uint64_t> node_index = node_logic_id(r);
    for (size_t i = 0; i < Dimension; ++i) {
      xr[r][i] = a[i] + node_index[i] * h[i];
    }
  }

  m_mesh = std::make_shared<Mesh<Connectivity<Dimension>>>(p_connectivity, xr);
}

template <>
void
CartesianMeshBuilder::_buildCartesianMesh(const TinyVector<3>& a,
                                          const TinyVector<3>& b,
                                          const TinyVector<3, uint64_t>& cell_size)
{
  constexpr size_t Dimension = 3;

  const TinyVector<Dimension, uint64_t> node_size = [&] {
    TinyVector node_size{cell_size};
    for (size_t i = 0; i < Dimension; ++i) {
      node_size[i] += 1;
    }
    return node_size;
  }();

  auto count_items = [](auto&& v) {
    using V_Type = std::decay_t<decltype(v)>;
    static_assert(is_tiny_vector_v<V_Type>);

    size_t sum = v[0];
    for (size_t i = 1; i < Dimension; ++i) {
      sum *= v[i];
    }
    return sum;
  };

  const size_t number_of_cells = count_items(cell_size);
  const size_t number_of_nodes = count_items(node_size);

  ConnectivityDescriptor descriptor;
  descriptor.node_number_vector.resize(number_of_nodes);
  for (size_t i = 0; i < number_of_nodes; ++i) {
    descriptor.node_number_vector[i] = i;
  }

  descriptor.cell_number_vector.resize(number_of_cells);
  for (size_t i = 0; i < number_of_cells; ++i) {
    descriptor.cell_number_vector[i] = i;
  }

  descriptor.cell_type_vector.resize(number_of_cells);
  std::fill(descriptor.cell_type_vector.begin(), descriptor.cell_type_vector.end(), CellType::Hexahedron);

  const auto cell_logic_id = [&](size_t j) {
    const size_t slice1  = cell_size[1] * cell_size[2];
    const size_t& slice2 = cell_size[2];
    const uint64_t j0    = j / slice1;
    const uint64_t j1    = (j - j0 * slice1) / slice2;
    const uint64_t j2    = j - (j0 * slice1 + j1 * slice2);
    return TinyVector<Dimension, uint64_t>{j0, j1, j2};
  };

  const auto node_logic_id = [&](size_t r) {
    const size_t slice1  = node_size[1] * node_size[2];
    const size_t& slice2 = node_size[2];
    const uint64_t r0    = r / slice1;
    const uint64_t r1    = (r - r0 * slice1) / slice2;
    const uint64_t r2    = r - (r0 * slice1 + r1 * slice2);
    return TinyVector<Dimension, uint64_t>{r0, r1, r2};
  };

  const auto node_number = [&](const TinyVector<Dimension, uint64_t>& node_logic_id) {
    return (node_logic_id[0] * node_size[1] + node_logic_id[1]) * node_size[2] + node_logic_id[2];
  };

  descriptor.cell_to_node_vector.resize(number_of_cells);
  constexpr size_t nb_node_per_cell = 1 << Dimension;
  for (size_t j = 0; j < number_of_cells; ++j) {
    TinyVector<Dimension, size_t> cell_index = cell_logic_id(j);
    descriptor.cell_to_node_vector[j].resize(nb_node_per_cell);
    for (size_t r = 0; r < nb_node_per_cell; ++r) {
      static_assert(Dimension == 3, "unexpected dimension");
      descriptor.cell_to_node_vector[j][0] = node_number(cell_index + TinyVector<Dimension, uint64_t>{0, 0, 0});
      descriptor.cell_to_node_vector[j][1] = node_number(cell_index + TinyVector<Dimension, uint64_t>{1, 0, 0});
      descriptor.cell_to_node_vector[j][2] = node_number(cell_index + TinyVector<Dimension, uint64_t>{1, 1, 0});
      descriptor.cell_to_node_vector[j][3] = node_number(cell_index + TinyVector<Dimension, uint64_t>{0, 1, 0});
      descriptor.cell_to_node_vector[j][4] = node_number(cell_index + TinyVector<Dimension, uint64_t>{0, 0, 1});
      descriptor.cell_to_node_vector[j][5] = node_number(cell_index + TinyVector<Dimension, uint64_t>{1, 0, 1});
      descriptor.cell_to_node_vector[j][6] = node_number(cell_index + TinyVector<Dimension, uint64_t>{1, 1, 1});
      descriptor.cell_to_node_vector[j][7] = node_number(cell_index + TinyVector<Dimension, uint64_t>{0, 1, 1});
    }
  }

  MeshBuilderBase::_computeCellFaceAndFaceNodeConnectivities<Dimension>(descriptor);
  MeshBuilderBase::_computeFaceEdgeAndEdgeNodeAndCellEdgeConnectivities<Dimension>(descriptor);

  this->_buildBoundaryNodeList(cell_size, descriptor);
  this->_buildBoundaryEdgeList(cell_size, descriptor);
  this->_buildBoundaryFaceList(cell_size, descriptor);

  descriptor.cell_owner_vector.resize(number_of_cells);
  std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

  descriptor.face_owner_vector.resize(descriptor.face_number_vector.size());
  std::fill(descriptor.face_owner_vector.begin(), descriptor.face_owner_vector.end(), parallel::rank());

  descriptor.edge_owner_vector.resize(descriptor.edge_number_vector.size());
  std::fill(descriptor.edge_owner_vector.begin(), descriptor.edge_owner_vector.end(), parallel::rank());

  descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
  std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

  std::shared_ptr p_connectivity        = Connectivity<Dimension>::build(descriptor);
  Connectivity<Dimension>& connectivity = *p_connectivity;

  const TinyVector<Dimension> h{(b[0] - a[0]) / cell_size[0], (b[1] - a[1]) / cell_size[1],
                                (b[2] - a[2]) / cell_size[2]};

  NodeValue<TinyVector<Dimension>> xr(connectivity);
  for (NodeId r = 0; r < number_of_nodes; ++r) {
    const TinyVector<Dimension, uint64_t> node_index = node_logic_id(r);
    for (size_t i = 0; i < Dimension; ++i) {
      xr[r][i] = a[i] + node_index[i] * h[i];
    }
  }

  m_mesh = std::make_shared<Mesh<Connectivity<Dimension>>>(p_connectivity, xr);
}

template <size_t Dimension>
CartesianMeshBuilder::CartesianMeshBuilder(const TinyVector<Dimension>& a,
                                           const TinyVector<Dimension>& b,
                                           const TinyVector<Dimension, uint64_t>& size)
{
  if (parallel::rank() == 0) {
    TinyVector lenght = b - a;
    for (size_t i = 0; i < Dimension; ++i) {
      if (lenght[i] == 0) {
        throw NormalError("invalid box definition corners share a component");
      }
    }

    TinyVector<Dimension> corner0 = a;
    TinyVector<Dimension> corner1 = b;

    for (size_t i = 0; i < Dimension; ++i) {
      if (corner0[i] > corner1[i]) {
        std::swap(corner0[i], corner1[i]);
      }
    }

    this->_buildCartesianMesh(corner0, corner1, size);
  }
  this->_dispatch<Dimension>();
}

template CartesianMeshBuilder::CartesianMeshBuilder(const TinyVector<1>&,
                                                    const TinyVector<1>&,
                                                    const TinyVector<1, uint64_t>&);

template CartesianMeshBuilder::CartesianMeshBuilder(const TinyVector<2>&,
                                                    const TinyVector<2>&,
                                                    const TinyVector<2, uint64_t>&);

template CartesianMeshBuilder::CartesianMeshBuilder(const TinyVector<3>&,
                                                    const TinyVector<3>&,
                                                    const TinyVector<3, uint64_t>&);
