#ifndef ACOUSTIC_SOLVER_HPP
#define ACOUSTIC_SOLVER_HPP

#include <memory>
#include <tuple>
#include <vector>

class IDiscreteFunction;
class IBoundaryConditionDescriptor;
class IMesh;

double acoustic_dt(const std::shared_ptr<const IDiscreteFunction>& c);

class AcousticSolverHandler
{
 public:
  enum class SolverType
  {
    Glace,
    Eucclhyd
  };

 private:
  struct IAcousticSolver
  {
    virtual std::tuple<std::shared_ptr<const IMesh>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>,
                       std::shared_ptr<const IDiscreteFunction>>
    apply(const double& dt,
          const std::shared_ptr<const IDiscreteFunction>& rho,
          const std::shared_ptr<const IDiscreteFunction>& u,
          const std::shared_ptr<const IDiscreteFunction>& E) const = 0;

    IAcousticSolver()                  = default;
    IAcousticSolver(IAcousticSolver&&) = default;
    IAcousticSolver& operator=(IAcousticSolver&&) = default;

    virtual ~IAcousticSolver() = default;
  };

  template <size_t Dimension>
  class AcousticSolver;

  std::unique_ptr<IAcousticSolver> m_acoustic_solver;

 public:
  const IAcousticSolver&
  solver() const
  {
    return *m_acoustic_solver;
  }

  AcousticSolverHandler(SolverType solver_type,
                        const std::shared_ptr<const IDiscreteFunction>& rho,
                        const std::shared_ptr<const IDiscreteFunction>& c,
                        const std::shared_ptr<const IDiscreteFunction>& u,
                        const std::shared_ptr<const IDiscreteFunction>& p,
                        const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>& bc_descriptor_list);
};

#endif   // ACOUSTIC_SOLVER_HPP
