#ifndef TABLE_HPP
#define TABLE_HPP

#include <utils/Array.hpp>
#include <utils/InvalidData.hpp>
#include <utils/PugsAssert.hpp>
#include <utils/PugsMacros.hpp>
#include <utils/PugsUtils.hpp>

#include <Kokkos_CopyViews.hpp>

#include <iostream>

template <typename DataType>
class [[nodiscard]] Table
{
 public:
  using data_type  = DataType;
  using index_type = size_t;

 private:
  Kokkos::View<DataType**> m_values;

  // Allows const version to access our data
  friend Table<std::add_const_t<DataType>>;

 public:
  PUGS_INLINE size_t numberOfRows() const noexcept
  {
    return m_values.extent(0);
  }

  PUGS_INLINE size_t numberOfColumns() const noexcept
  {
    return m_values.extent(1);
  }

  PUGS_INLINE
  Array<DataType> operator[](index_type i) const
  {
    Assert(i < this->numberOfRows());
    return encapsulate(Kokkos::View<DataType*>(m_values, i, Kokkos::ALL));
  }

  friend PUGS_INLINE Table<std::remove_const_t<DataType>> copy(const Table<DataType>& source)
  {
    Table<std::remove_const_t<DataType>> image(source.numberOfRows(), source.numberOfColumns());
    Kokkos::deep_copy(image.m_values, source.m_values);

    return image;
  }

  friend PUGS_INLINE void copy_to(const Table<DataType>& source,
                                  const Table<std::remove_const_t<DataType>>& destination)
  {
    Assert(source.numberOfRows() == destination.numberOfRows());
    Assert(source.numberOfColumns() == destination.numberOfColumns());
    Kokkos::deep_copy(destination.m_values, source.m_values);
  }

  template <typename DataType2, typename... RT>
  friend PUGS_INLINE Table<DataType2> encapsulate(const Kokkos::View<DataType2**, RT...>& values);

  template <typename DataType2>
  friend PUGS_INLINE Table<DataType2> subTable(const Table<DataType2>& table,
                                               typename Table<DataType2>::index_type row_begin,
                                               typename Table<DataType2>::index_type row_size,
                                               typename Table<DataType2>::index_type column_begin,
                                               typename Table<DataType2>::index_type column_size);

  PUGS_INLINE DataType& operator()(index_type i, index_type j) const noexcept(NO_ASSERT)
  {
    Assert(i < this->numberOfRows());
    Assert(j < this->numberOfColumns());
    return m_values(i, j);
  }

  PUGS_INLINE
  void fill(const DataType& data) const
  {
    static_assert(not std::is_const<DataType>(), "Cannot modify Table of const");

    Kokkos::deep_copy(m_values, data);
  }

  template <typename DataType2>
  PUGS_INLINE Table& operator=(const Table<DataType2>& table) noexcept
  {
    // ensures that DataType is the same as source DataType2
    static_assert(std::is_same<std::remove_const_t<DataType>, std::remove_const_t<DataType2>>(),
                  "Cannot assign Table of different type");
    // ensures that const is not lost through copy
    static_assert(((std::is_const<DataType2>() and std::is_const<DataType>()) or not std::is_const<DataType2>()),
                  "Cannot assign Table of const to Table of non-const");
    m_values = table.m_values;
    return *this;
  }

  PUGS_INLINE
  Table& operator=(const Table&) = default;

  PUGS_INLINE
  Table& operator=(Table&&) = default;

  PUGS_INLINE
  explicit Table(size_t nb_lines, size_t nb_columns)
  {
    static_assert(not std::is_const<DataType>(), "Cannot allocate Table of const data: only view is "
                                                 "supported");
    if constexpr (std::is_arithmetic_v<DataType>) {
      m_values =
        Kokkos::View<DataType**>{Kokkos::view_alloc(Kokkos::WithoutInitializing, "anonymous"), nb_lines, nb_columns};
    } else {
      m_values = Kokkos::View<DataType**>{"anonymous", nb_lines, nb_columns};
    }

#ifndef NDEBUG
    if constexpr (not std::is_const_v<DataType>) {
      using T = std::decay_t<DataType>;
      if constexpr (std::is_arithmetic_v<T>) {
        this->fill(invalid_data_v<T>);
      } else if constexpr (is_tiny_vector_v<T>) {
        this->fill(T{});
      } else if constexpr (is_tiny_matrix_v<T>) {
        this->fill(T{});
      }
    }
#endif   // NDEBUG
  }

  friend std::ostream& operator<<(std::ostream& os, const Table& t)
  {
    for (size_t i = 0; i < t.numberOfRows(); ++i) {
      os << i << '|';
      for (size_t j = 0; j < t.numberOfColumns(); ++j) {
        os << ' ' << j << ':' << NaNHelper(t(i, j));
      }
      os << '\n';
    }
    return os;
  }

  PUGS_INLINE
  Table() = default;

  PUGS_INLINE
  Table(const Table&) = default;

  template <typename DataType2>
  PUGS_INLINE Table(const Table<DataType2>& table) noexcept
  {
    this->operator=(table);
  }

  PUGS_INLINE
  Table(Table &&) = default;

  PUGS_INLINE
  ~Table() = default;
};

template <typename DataType, typename... RT>
PUGS_INLINE Table<DataType>
encapsulate(const Kokkos::View<DataType**, RT...>& values)
{
  Table<DataType> table;
  table.m_values = values;
  return table;
}

template <typename DataType>
PUGS_INLINE Table<DataType>
subTable(const Table<DataType>& table,
         typename Table<DataType>::index_type row_begin,
         typename Table<DataType>::index_type row_size,
         typename Table<DataType>::index_type column_begin,
         typename Table<DataType>::index_type column_size)
{
  Assert(row_begin < table.numberOfRows());
  Assert(row_begin + row_size <= table.numberOfRows());
  Assert(column_begin < table.numberOfColumns());
  Assert(column_begin + column_size <= table.numberOfColumns());
  return encapsulate(Kokkos::View<DataType**>(table.m_values, std::make_pair(row_begin, row_begin + row_size),
                                              std::make_pair(column_begin, column_begin + column_size)));
}

#endif   // TABLE_HPP
