#ifndef POLYNOMIALP_HPP
#define POLYNOMIALP_HPP

#include <algebra/TinyVector.hpp>
#include <utils/Exceptions.hpp>

template <size_t N, size_t Dimension>
class PolynomialP
{
 private:
  static constexpr size_t size_coef = [] {
    if constexpr (Dimension == 1) {
      return N + 1;
    } else if constexpr (Dimension == 2) {
      return (N + 1) * (N + 2) / 2;
    } else {
      static_assert(Dimension == 3);
      return (N + 1) * (N + 2) * (N + 3) / 6;
    }
  }();

  using Coefficients = TinyVector<size_coef, double>;
  Coefficients m_coefficients;
  static_assert((N >= 0), "PolynomialP degree must be non-negative");
  static_assert((Dimension > 0), "PolynomialP dimension must be positive");
  static_assert((Dimension <= 3), "PolynomialP dimension must no greater than three");

 public:
  PUGS_INLINE
  constexpr size_t
  degree() const
  {
    return N;
  }

  constexpr size_t
  dim() const
  {
    return Dimension;
  }

  PUGS_INLINE
  constexpr const TinyVector<size_coef, double>&
  coefficients() const
  {
    return m_coefficients;
  }

  PUGS_INLINE
  constexpr TinyVector<size_coef, double>&
  coefficients()
  {
    return m_coefficients;
  }

  PUGS_INLINE constexpr bool
  operator==(const PolynomialP& q) const
  {
    return m_coefficients == q.m_coefficients;
  }

  PUGS_INLINE constexpr PolynomialP(const TinyVector<size_coef, double>& coefficients) noexcept
    : m_coefficients{coefficients}
  {}

  PUGS_INLINE
  constexpr PolynomialP(TinyVector<size_coef, double>&& coefficients) noexcept : m_coefficients{coefficients} {}

  PUGS_INLINE
  constexpr bool
  operator!=(const PolynomialP& q) const
  {
    return not this->operator==(q);
  }

  PUGS_INLINE constexpr PolynomialP
  operator+(const PolynomialP Q) const
  {
    PolynomialP<N, Dimension> P(m_coefficients);
    for (size_t i = 0; i < size_coef; ++i) {
      P.coefficients()[i] += Q.coefficients()[i];
    }
    return P;
  }

  PUGS_INLINE constexpr PolynomialP
  operator-() const
  {
    PolynomialP<N, Dimension> P;
    P.coefficients() = -coefficients();
    return P;
  }

  PUGS_INLINE constexpr PolynomialP
  operator-(const PolynomialP Q) const
  {
    PolynomialP<N, Dimension> P(m_coefficients);
    P = P + (-Q);
    return P;
  }

  template <size_t M, size_t Dim>
  PUGS_INLINE constexpr PolynomialP&
  operator=(const PolynomialP<M, Dim>& Q)
  {
    coefficients() = zero;
    for (size_t i = 0; i < size_coef; ++i) {
      coefficients()[i] = Q.coefficients()[i];
    }
    return *this;
  }

  PUGS_INLINE constexpr PolynomialP&
  operator+=(const PolynomialP& Q)
  {
    m_coefficients += Q.coefficients();
    return *this;
  }

  template <size_t M>
  PUGS_INLINE constexpr PolynomialP&
  operator-=(const PolynomialP& Q)
  {
    m_coefficients -= Q.coefficients();
    return *this;
  }

  PUGS_INLINE
  constexpr PolynomialP
  operator*(const double& lambda) const
  {
    TinyVector<size_coef> mult_coefs = lambda * m_coefficients;
    PolynomialP<N, Dimension> M(mult_coefs);
    return M;
  }

  PUGS_INLINE
  constexpr friend PolynomialP<N, Dimension>
  operator*(const double& lambda, const PolynomialP<N, Dimension> P)
  {
    return P * lambda;
  }

  PUGS_INLINE
  constexpr double
  operator[](const TinyVector<Dimension, size_t> relative_pos) const
  {
    size_t total_degree = 0;
    for (size_t i = 0; i < Dimension; ++i) {
      Assert((relative_pos[i] <= N),
             "You are looking for a coefficient of order greater than the degree of the polynomial");
      total_degree += relative_pos[i];
    }
    Assert((total_degree <= N), "The sum of the degrees of the coefficient you are looking for is greater than the "
                                "degree of the polynomial");
    TinyVector<size_coef> absolute_coefs = this->coefficients();
    size_t absolute_position             = 0;
    if constexpr (Dimension == 1) {
      absolute_position = relative_pos[0];
    } else if constexpr (Dimension == 2) {
      size_t total_degree = relative_pos[0] + relative_pos[1];
      absolute_position   = total_degree * (total_degree + 1) / 2 + relative_pos[1];
    } else {
      throw NotImplementedError("Not yet Available in 3D");
      // static_assert(Dimension == 3);
      // size_t total_degree = relative_pos[0] + relative_pos[1] + relative_pos[2];
      // return (total_degree + 1) * (total_degree + 2) * (total_degree + 3) / 6 + relative_pos[1];
      // return (N + 1) * (N + 2) * (N + 3) / 6;
    }

    return absolute_coefs[absolute_position];
  }

  PUGS_INLINE
  constexpr double
  operator[](const TinyVector<Dimension, size_t> relative_pos)
  {
    size_t total_degree = 0;
    for (size_t i = 0; i < Dimension; ++i) {
      Assert((relative_pos[i] <= N),
             "You are looking for a coefficient of order greater than the degree of the polynomial");
      total_degree += relative_pos[i];
    }
    Assert((total_degree <= N), "The sum of the degrees of the coefficient you are looking for is greater than the "
                                "degree of the polynomial");
    TinyVector<size_coef> absolute_coefs = this->coefficients();
    size_t absolute_position             = 0;
    if constexpr (Dimension == 1) {
      absolute_position = relative_pos[0];
    } else if constexpr (Dimension == 2) {
      size_t total_degree = relative_pos[0] + relative_pos[1];
      absolute_position   = total_degree * (total_degree + 1) / 2 + relative_pos[1];
    } else {
      throw NotImplementedError("Not yet Available in 3D");
      // static_assert(Dimension == 3);
      // size_t total_degree = relative_pos[0] + relative_pos[1] + relative_pos[2];
      // return (total_degree + 1) * (total_degree + 2) * (total_degree + 3) / 6 + relative_pos[1];
      // return (N + 1) * (N + 2) * (N + 3) / 6;
    }

    return absolute_coefs[absolute_position];
  }

  PUGS_INLINE
  constexpr double
  absolute_position(const TinyVector<Dimension, size_t> relative_pos) const
  {
    size_t total_degree = 0;
    for (size_t i = 0; i < Dimension; ++i) {
      Assert((relative_pos[i] <= N),
             "You are looking for a coefficient of order greater than the degree of the polynomial");
      total_degree += relative_pos[i];
    }
    Assert((total_degree <= N), "The sum of the degrees of the coefficient you are looking for is greater than the "
                                "degree of the polynomial");
    size_t abs_pos = 0;
    if constexpr (Dimension == 1) {
      abs_pos = relative_pos[0];
    } else if constexpr (Dimension == 2) {
      size_t total_degree = relative_pos[0] + relative_pos[1];
      abs_pos             = total_degree * (total_degree + 1) / 2 + relative_pos[1];
    } else {
      throw NotImplementedError("Not yet Available in 3D");
      // static_assert(Dimension == 3);
      // size_t total_degree = relative_pos[0] + relative_pos[1] + relative_pos[2];
      // return (total_degree + 1) * (total_degree + 2) * (total_degree + 3) / 6 + relative_pos[1];
      // return (N + 1) * (N + 2) * (N + 3) / 6;
    }

    return abs_pos;
  }

  PUGS_INLINE
  constexpr double
  operator()(const TinyVector<Dimension> x) const
  {
    const auto& P = *this;
    double value  = 0.;
    if constexpr (Dimension == 1) {
      value = m_coefficients[N];
      for (size_t i = N; i > 0; --i) {
        value *= x;
        value += m_coefficients[i - 1];
      }
    } else if constexpr (Dimension == 2) {
      TinyVector<Dimension, size_t> relative_pos(0, N);
      value = P[relative_pos];
      for (size_t i = N; i > 0; --i) {
        value *= x[1];
        relative_pos  = TinyVector<Dimension, size_t>(N - i + 1, i - 1);
        double valuex = P[relative_pos];
        for (size_t j = N - i + 1; j > 0; --j) {
          valuex *= x[0];
          relative_pos = TinyVector<Dimension, size_t>(j - 1, i - 1);
          valuex += P[relative_pos];
        }
        value += valuex;
      }
    } else {
      throw NotImplementedError("Not yet Available in 3D");
    }

    return value;
  }

  PUGS_INLINE size_t
  find_size_coef(const size_t degree)
  {
    if constexpr (Dimension == 1) {
      return degree + 1;
    } else if constexpr (Dimension == 2) {
      return (degree + 1) * (degree + 2) / 2;
    } else {
      static_assert(Dimension == 3);
      return (degree + 1) * (degree + 2) * (degree + 3) / 6;
    }
  }

  PUGS_INLINE constexpr auto
  derivative(const size_t var) const
  {
    const auto P = *this;
    TinyVector<size_coef> coefs(zero);
    PolynomialP<N, Dimension> Q(coefs);
    if constexpr (N == 0) {
      return Q;
    } else {
      Assert(var < Dimension,
             "You can not derive a polynomial with respect to a variable of rank greater than the dimension");
      if constexpr (Dimension == 1) {
        for (size_t i = 0; i < size_coef; ++i) {
          coefs[i] = double(i + 1) * P.coefficients()[i + 1];
        }
        return Q;
      } else if constexpr (Dimension == 2) {
        if (var == 0) {
          for (size_t i = 0; i < N; ++i) {
            for (size_t j = 0; j < N - i; ++j) {
              TinyVector<Dimension, size_t> relative_pos(i, j);
              TinyVector<Dimension, size_t> relative_posp(i + 1, j);
              size_t absolute_position            = Q.absolute_position(relative_pos);
              size_t absolute_positionp           = P.absolute_position(relative_posp);
              Q.coefficients()[absolute_position] = double(i + 1) * m_coefficients[absolute_positionp];
            }
          }
        } else {
          for (size_t i = 0; i < N; ++i) {
            for (size_t j = 0; j < N - i; ++j) {
              TinyVector<Dimension, size_t> relative_pos(i, j);
              TinyVector<Dimension, size_t> relative_posp(i, j + 1);
              size_t absolute_position            = Q.absolute_position(relative_pos);
              size_t absolute_positionp           = P.absolute_position(relative_posp);
              Q.coefficients()[absolute_position] = double(j + 1) * m_coefficients[absolute_positionp];
            }
          }
        }
        return Q;
      } else {
        throw UnexpectedError("Not yet Available in 3D");
      }
    }
  }

  PUGS_INLINE constexpr PolynomialP() noexcept = default;
  ~PolynomialP()                               = default;
};

//   template <size_t M>
//   PUGS_INLINE constexpr friend void
//   divide(const PolynomialP<N>& P1, const PolynomialP<M>& P2, PolynomialP<N>& Q, PolynomialP<N>& R)
//   {
//     const size_t Nr  = P1.realDegree();
//     const size_t Mr  = P2.realDegree();
//     R.coefficients() = P1.coefficients();
//     Q.coefficients() = zero;
//     for (ssize_t k = Nr - Mr; k >= 0; --k) {
//       Q.coefficients()[k] = R.coefficients()[Mr + k] / P2.coefficients()[Mr];
//       for (ssize_t j = Mr + k; j >= k; --j) {
//         R.coefficients()[j] -= Q.coefficients()[k] * P2.coefficients()[j - k];
//       }
//     }
//     for (size_t j = Mr; j <= Nr; ++j) {
//       R.coefficients()[j] = 0;
//     }
//   }

//   PUGS_INLINE
//   constexpr friend PolynomialP<N + 1>
//   primitive(const PolynomialP<N>& P)
//   {
//     TinyVector<N + 2> coefs;
//     for (size_t i = 0; i < N + 1; ++i) {
//       coefs[i + 1] = P.coefficients()[i] / double(i + 1);
//     }
//     coefs[0] = 0;
//     return PolynomialP<N + 1>{coefs};
//   }

//   PUGS_INLINE
//   constexpr friend std::ostream&
//   operator<<(std::ostream& os, const PolynomialP<N>& P)
//   {
//     //    os << "P(x) = ";
//     bool all_coef_zero = true;
//     if (N == 0) {
//       os << P.coefficients()[0];
//       return os;
//     }
//     if (N != 1) {
//       if (P.coefficients()[N] != 0.) {
//         if (P.coefficients()[N] < 0.) {
//           os << "- ";
//         }
//         if (P.coefficients()[N] != 1 && P.coefficients()[N] != -1) {
//           os << std::abs(P.coefficients()[N]);
//         }
//         os << "x^" << N;
//         all_coef_zero = false;
//       }
//     }
//     for (size_t i = N - 1; i > 1; --i) {
//       if (P.coefficients()[i] != 0.) {
//         if (P.coefficients()[i] > 0.) {
//           os << " + ";
//         } else if (P.coefficients()[i] < 0.) {
//           os << " - ";
//         }
//         if (P.coefficients()[i] != 1 && P.coefficients()[i] != -1) {
//           os << std::abs(P.coefficients()[i]);
//         }
//         os << "x^" << i;
//         all_coef_zero = false;
//       }
//     }
//     if (P.coefficients()[1] != 0.) {
//       if (P.coefficients()[1] > 0. && N != 1) {
//         os << " + ";
//       } else if (P.coefficients()[1] < 0.) {
//         os << " - ";
//       }
//       if (P.coefficients()[1] != 1 && P.coefficients()[1] != -1) {
//         os << std::abs(P.coefficients()[1]);
//       }
//       os << "x";
//       all_coef_zero = false;
//     }
//     if (P.coefficients()[0] != 0. || all_coef_zero) {
//       if (P.coefficients()[0] > 0.) {
//         os << " + ";
//       } else if (P.coefficients()[0] < 0.) {
//         os << " - ";
//       }
//       os << std::abs(P.coefficients()[0]);
//     }
//     return os;
//   }

//   PUGS_INLINE
//   constexpr friend void
//   lagrangeBasis(const TinyVector<N + 1> zeros, TinyVector<N + 1, PolynomialP<N>>& basis)
//   {
//     PolynomialP<N> lj;
//     for (size_t j = 0; j < N + 1; ++j) {
//       basis[j] = lagrangePolynomialP(zeros, j);
//     }
//   }

//   PUGS_INLINE
//   constexpr friend PolynomialP<N>
//   lagrangeToCanonical(const TinyVector<N + 1> lagrange_coefs, const TinyVector<N + 1, PolynomialP<N>>& basis)
//   {
//     PolynomialP<N> P(zero);
//     //    lagrangeBasis({0, 0, 0}, basis);
//     for (size_t j = 0; j < N + 1; ++j) {
//       P += basis[j] * lagrange_coefs[j];
//     }
//     return P;
//   }

// template <size_t N>
// PUGS_INLINE constexpr PolynomialP<N> lagrangePolynomialP(const TinyVector<N + 1> zeros, const size_t k);

// template <size_t N>
// PUGS_INLINE constexpr TinyVector<N, PolynomialP<N - 1>>
// lagrangeBasis(const TinyVector<N>& zeros)
// {
//   static_assert(N >= 1, "invalid degree");
//   TinyVector<N, PolynomialP<N - 1>> basis;
//   PolynomialP<N - 1> lj;
//   for (size_t j = 0; j < N; ++j) {
//     basis[j] = lagrangePolynomialP<N - 1>(zeros, j);
//   }
//   return basis;
// }

// template <size_t N>
// PUGS_INLINE constexpr double
// integrate(const PolynomialP<N>& P, const double& xinf, const double& xsup)
// {
//   PolynomialP<N + 1> Q = primitive(P);
//   return (Q(xsup) - Q(xinf));
// }

// template <size_t N>
// PUGS_INLINE constexpr double
// symmetricIntegrate(const PolynomialP<N>& P, const double& delta)
// {
//   Assert(delta > 0, "A positive delta is needed for symmetricIntegrate");
//   double integral = 0.;
//   for (size_t i = 0; i <= N; ++i) {
//     if (i % 2 == 0)
//       integral += 2. * P.coefficients()[i] * std::pow(delta, i + 1) / (i + 1);
//   }
//   return integral;
// }

// template <size_t N>
// PUGS_INLINE constexpr PolynomialP<N>
// lagrangePolynomialP(const TinyVector<N + 1> zeros, const size_t k)
// {
//   for (size_t i = 0; i < N; ++i) {
//     Assert(zeros[i] < zeros[i + 1], "Interpolation values must be strictly increasing in Lagrange polynomialPs");
//   }
//   PolynomialP<N> lk;
//   lk.coefficients()    = zero;
//   lk.coefficients()[0] = 1;
//   for (size_t i = 0; i < N + 1; ++i) {
//     if (k == i)
//       continue;
//     double factor = 1. / (zeros[k] - zeros[i]);
//     PolynomialP<1> P({-zeros[i] * factor, factor});
//     lk *= P;
//   }
//   return lk;
// }

#endif   // POLYNOMIALP_HPP
