#include <catch2/catch_test_macros.hpp>
#include <catch2/matchers/catch_matchers_all.hpp>

#include <MeshDataBaseForTests.hpp>

#include <language/utils/EmbeddedIDiscreteFunctionMathFunctions.hpp>
#include <scheme/DiscreteFunctionP0.hpp>
#include <scheme/DiscreteFunctionP0Vector.hpp>

// clazy:excludeall=non-pod-global-static

#define CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(P_U, FCT)          \
  {                                                                         \
    using DiscreteFunctionType = const std::decay_t<decltype(*P_U)>;        \
    std::shared_ptr p_fu       = ::FCT(P_U);                                \
                                                                            \
    REQUIRE(p_fu.use_count() > 0);                                          \
    REQUIRE_NOTHROW(dynamic_cast<const DiscreteFunctionType&>(*p_fu));      \
                                                                            \
    const auto& fu = dynamic_cast<const DiscreteFunctionType&>(*p_fu);      \
                                                                            \
    auto values  = P_U->cellValues();                                       \
    bool is_same = true;                                                    \
    for (CellId cell_id = 0; cell_id < values.numberOfItems(); ++cell_id) { \
      if (fu[cell_id] != std::FCT(values[cell_id])) {                       \
        is_same = false;                                                    \
        break;                                                              \
      }                                                                     \
    }                                                                       \
                                                                            \
    REQUIRE(is_same);                                                       \
  }

#define CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(P_LHS, P_RHS, FCT)             \
  {                                                                                 \
    using DiscreteFunctionType = const std::decay_t<decltype(FCT(*P_LHS, *P_RHS))>; \
    std::shared_ptr p_fuv      = ::FCT(P_LHS, P_RHS);                               \
                                                                                    \
    REQUIRE(p_fuv.use_count() > 0);                                                 \
    REQUIRE_NOTHROW(dynamic_cast<const DiscreteFunctionType&>(*p_fuv));             \
                                                                                    \
    const auto& fuv = dynamic_cast<const DiscreteFunctionType&>(*p_fuv);            \
                                                                                    \
    auto lhs_values = P_LHS->cellValues();                                          \
    auto rhs_values = P_RHS->cellValues();                                          \
    bool is_same    = true;                                                         \
    for (CellId cell_id = 0; cell_id < lhs_values.numberOfItems(); ++cell_id) {     \
      using namespace std;                                                          \
      if (fuv[cell_id] != FCT(lhs_values[cell_id], rhs_values[cell_id])) {          \
        is_same = false;                                                            \
        break;                                                                      \
      }                                                                             \
    }                                                                               \
                                                                                    \
    REQUIRE(is_same);                                                               \
  }

#define CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(P_LHS, RHS, FCT)           \
  {                                                                              \
    using DiscreteFunctionType = const std::decay_t<decltype(FCT(*P_LHS, RHS))>; \
    std::shared_ptr p_fuv      = ::FCT(P_LHS, RHS);                              \
                                                                                 \
    REQUIRE(p_fuv.use_count() > 0);                                              \
    REQUIRE_NOTHROW(dynamic_cast<const DiscreteFunctionType&>(*p_fuv));          \
                                                                                 \
    const auto& fuv = dynamic_cast<const DiscreteFunctionType&>(*p_fuv);         \
                                                                                 \
    auto lhs_values = P_LHS->cellValues();                                       \
    bool is_same    = true;                                                      \
    for (CellId cell_id = 0; cell_id < lhs_values.numberOfItems(); ++cell_id) {  \
      using namespace std;                                                       \
      if (fuv[cell_id] != FCT(lhs_values[cell_id], RHS)) {                       \
        is_same = false;                                                         \
        break;                                                                   \
      }                                                                          \
    }                                                                            \
                                                                                 \
    REQUIRE(is_same);                                                            \
  }

#define CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(LHS, P_RHS, FCT)           \
  {                                                                              \
    using DiscreteFunctionType = const std::decay_t<decltype(FCT(LHS, *P_RHS))>; \
    std::shared_ptr p_fuv      = ::FCT(LHS, P_RHS);                              \
                                                                                 \
    REQUIRE(p_fuv.use_count() > 0);                                              \
    REQUIRE_NOTHROW(dynamic_cast<const DiscreteFunctionType&>(*p_fuv));          \
                                                                                 \
    const auto& fuv = dynamic_cast<const DiscreteFunctionType&>(*p_fuv);         \
                                                                                 \
    auto rhs_values = P_RHS->cellValues();                                       \
    bool is_same    = true;                                                      \
    for (CellId cell_id = 0; cell_id < rhs_values.numberOfItems(); ++cell_id) {  \
      using namespace std;                                                       \
      if (fuv[cell_id] != FCT(LHS, rhs_values[cell_id])) {                       \
        is_same = false;                                                         \
        break;                                                                   \
      }                                                                          \
    }                                                                            \
                                                                                 \
    REQUIRE(is_same);                                                            \
  }

TEST_CASE("EmbeddedIDiscreteFunctionMathFunctions", "[scheme]")
{
  SECTION("1D")
  {
    constexpr size_t Dimension = 1;

    using Rd = TinyVector<Dimension>;

    std::array mesh_list = MeshDataBaseForTests::get().all1DMeshes();

    for (auto named_mesh : mesh_list) {
      SECTION(named_mesh.name())
      {
        auto mesh = named_mesh.mesh();

        std::shared_ptr other_mesh =
          std::make_shared<Mesh<Connectivity<Dimension>>>(mesh->shared_connectivity(), mesh->xr());

        CellValue<const Rd> xj = MeshDataManager::instance().getMeshData(*mesh).xj();

        CellValue<double> values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 0.2 + std::cos(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        CellValue<double> positive_values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 2 + std::sin(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        CellValue<double> bounded_values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 0.9 * std::sin(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        std::shared_ptr p_u = std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, values);
        std::shared_ptr p_other_mesh_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(other_mesh, values);
        std::shared_ptr p_positive_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, positive_values);
        std::shared_ptr p_bounded_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, bounded_values);

        std::shared_ptr p_R1_u = [=] {
          CellValue<TinyVector<1>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) { uj[cell_id][0] = 2 * xj[cell_id][0] + 1; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(mesh, uj);
        }();

        std::shared_ptr p_R1_v = [=] {
          CellValue<TinyVector<1>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { vj[cell_id][0] = xj[cell_id][0] * xj[cell_id][0] + 1; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R1_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(other_mesh, p_R1_u->cellValues());

        constexpr auto to_2d = [&](const TinyVector<Dimension>& x) -> TinyVector<2> {
          if constexpr (Dimension == 1) {
            return TinyVector<2>{x[0], 1 + x[0] * x[0]};
          } else if constexpr (Dimension == 2) {
            return TinyVector<2>{x[0], x[1]};
          } else if constexpr (Dimension == 3) {
            return TinyVector<2>{x[0], x[1] + x[2]};
          }
        };

        std::shared_ptr p_R2_u = [=] {
          CellValue<TinyVector<2>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);
              uj[cell_id]           = TinyVector<2>{2 * x[0] + 1, 1 - x[1]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(mesh, uj);
        }();

        std::shared_ptr p_R2_v = [=] {
          CellValue<TinyVector<2>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);
              vj[cell_id]           = TinyVector<2>{x[0] * x[1] + 1, 2 * x[1]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R2_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(other_mesh, p_R2_u->cellValues());

        constexpr auto to_3d = [&](const TinyVector<Dimension>& x) -> TinyVector<3> {
          if constexpr (Dimension == 1) {
            return TinyVector<3>{x[0], 1 + x[0] * x[0], 2 - x[0]};
          } else if constexpr (Dimension == 2) {
            return TinyVector<3>{x[0], x[1], x[0] + x[1]};
          } else if constexpr (Dimension == 3) {
            return TinyVector<3>{x[0], x[1], x[2]};
          }
        };

        std::shared_ptr p_R3_u = [=] {
          CellValue<TinyVector<3>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              uj[cell_id]           = TinyVector<3>{2 * x[0] + 1, 1 - x[1] * x[2], x[0] + x[2]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(mesh, uj);
        }();

        std::shared_ptr p_R3_v = [=] {
          CellValue<TinyVector<3>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              vj[cell_id]           = TinyVector<3>{x[0] * x[1] + 1, 2 * x[1], x[2] * x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R3_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(other_mesh, p_R3_u->cellValues());

        std::shared_ptr p_R1x1_u = [=] {
          CellValue<TinyMatrix<1>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { uj[cell_id] = TinyMatrix<1>{2 * xj[cell_id][0] + 1}; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>>(mesh, uj);
        }();

        std::shared_ptr p_R2x2_u = [=] {
          CellValue<TinyMatrix<2>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);

              uj[cell_id] = TinyMatrix<2>{2 * x[0] + 1, 1 - x[1],   //
                                          2 * x[1], -x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>>(mesh, uj);
        }();

        std::shared_ptr p_R3x3_u = [=] {
          CellValue<TinyMatrix<3>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);

              uj[cell_id] = TinyMatrix<3>{2 * x[0] + 1,    1 - x[1],        3,             //
                                          2 * x[1],        -x[0],           x[0] - x[1],   //
                                          3 * x[2] - x[1], x[1] - 2 * x[2], x[2] - x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>>(mesh, uj);
        }();

        std::shared_ptr p_Vector3_u = [=] {
          CellArray<double> uj_vector{mesh->connectivity(), 3};
          parallel_for(
            uj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              uj_vector[cell_id][0] = 2 * x[0] + 1;
              uj_vector[cell_id][1] = 1 - x[1] * x[2];
              uj_vector[cell_id][2] = x[0] + x[2];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, uj_vector);
        }();

        std::shared_ptr p_Vector3_v = [=] {
          CellArray<double> vj_vector{mesh->connectivity(), 3};
          parallel_for(
            vj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              vj_vector[cell_id][0] = x[0] * x[1] + 1;
              vj_vector[cell_id][1] = 2 * x[1];
              vj_vector[cell_id][2] = x[2] * x[0];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, vj_vector);
        }();

        std::shared_ptr p_Vector2_w = [=] {
          CellArray<double> wj_vector{mesh->connectivity(), 2};
          parallel_for(
            wj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              wj_vector[cell_id][0] = x[0] + x[1] * 2;
              wj_vector[cell_id][1] = x[0] * x[1];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, wj_vector);
        }();

        SECTION("sqrt Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, sqrt);
          REQUIRE_THROWS_WITH(sqrt(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("abs Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, abs);
          REQUIRE_THROWS_WITH(abs(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("sin Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, sin);
          REQUIRE_THROWS_WITH(sin(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("cos Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, cos);
          REQUIRE_THROWS_WITH(cos(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("tan Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, tan);
          REQUIRE_THROWS_WITH(tan(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("asin Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, asin);
          REQUIRE_THROWS_WITH(asin(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("acos Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, acos);
          REQUIRE_THROWS_WITH(acos(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atan Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, atan);
          REQUIRE_THROWS_WITH(atan(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("sinh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, sinh);
          REQUIRE_THROWS_WITH(sinh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("cosh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, cosh);
          REQUIRE_THROWS_WITH(cosh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("tanh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, tanh);
          REQUIRE_THROWS_WITH(tanh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("asinh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, asinh);
          REQUIRE_THROWS_WITH(asinh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("acosh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, acosh);
          REQUIRE_THROWS_WITH(acosh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atanh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, atanh);
          REQUIRE_THROWS_WITH(atanh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("exp Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, exp);
          REQUIRE_THROWS_WITH(exp(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("log Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, log);
          REQUIRE_THROWS_WITH(log(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atan2 Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_positive_u, p_bounded_u, atan2);
          REQUIRE_THROWS_WITH(atan2(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(atan2(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
          REQUIRE_THROWS_WITH(atan2(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
        }

        SECTION("atan2 Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 3.6, atan2);
          REQUIRE_THROWS_WITH(atan2(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("atan2 R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(2.4, p_u, atan2);
          REQUIRE_THROWS_WITH(atan2(2.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("min Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_u, p_bounded_u, min);
          REQUIRE_THROWS_WITH(::min(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(::min(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(::min(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("min Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 1.2, min);
          REQUIRE_THROWS_WITH(min(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("min R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(0.4, p_u, min);
          REQUIRE_THROWS_WITH(min(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("min Vh -> R")
        {
          REQUIRE(min(std::shared_ptr<const IDiscreteFunction>{p_u}) == min(p_u->cellValues()));
          REQUIRE_THROWS_WITH(min(std::shared_ptr<const IDiscreteFunction>{p_R1_u}),
                              "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("max Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_u, p_bounded_u, max);
          REQUIRE_THROWS_WITH(::max(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(::max(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(::max(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("max Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 1.2, max);
          REQUIRE_THROWS_WITH(max(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("max Vh -> R")
        {
          REQUIRE(max(std::shared_ptr<const IDiscreteFunction>{p_u}) == max(p_u->cellValues()));
          REQUIRE_THROWS_WITH(max(std::shared_ptr<const IDiscreteFunction>{p_R1_u}),
                              "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("max R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(0.4, p_u, max);
          REQUIRE_THROWS_WITH(max(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("pow Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_positive_u, p_bounded_u, pow);
          REQUIRE_THROWS_WITH(pow(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(pow(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(pow(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("pow Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_positive_u, 3.3, pow);
          REQUIRE_THROWS_WITH(pow(p_R1_u, 3.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("pow R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(2.1, p_u, pow);
          REQUIRE_THROWS_WITH(pow(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("dot Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R1_u, p_R1_v, dot);
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R2_u, p_R2_v, dot);
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R3_u, p_R3_v, dot);

          {
            auto p_UV = dot(p_Vector3_u, p_Vector3_v);
            REQUIRE(p_UV.use_count() == 1);

            auto UV        = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*p_UV);
            auto direct_UV = dot(*p_Vector3_u, *p_Vector3_v);

            bool is_same = true;
            for (CellId cell_id = 0; cell_id < mesh->numberOfCells(); ++cell_id) {
              if (UV[cell_id] != direct_UV[cell_id]) {
                is_same = false;
                break;
              }
            }

            REQUIRE(is_same);
          }

          REQUIRE_THROWS_WITH(dot(p_R1_u, p_other_mesh_R1_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R2_u, p_other_mesh_R2_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R3_u, p_other_mesh_R3_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R1_u, p_R3_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(dot(p_Vector3_u, p_Vector2_w), "error: operands have different dimension");
        }

        SECTION("dot Vh*Rd -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R1_u, (TinyVector<1>{3}), dot);
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R2_u, (TinyVector<2>{-6, 2}), dot);
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R3_u, (TinyVector<3>{-1, 5, 2}), dot);
          REQUIRE_THROWS_WITH(dot(p_R1_u, (TinyVector<2>{-6, 2})),
                              "error: incompatible operand types Vh(P0:R^1) and R^2");
          REQUIRE_THROWS_WITH(dot(p_R2_u, (TinyVector<3>{-1, 5, 2})),
                              "error: incompatible operand types Vh(P0:R^2) and R^3");
          REQUIRE_THROWS_WITH(dot(p_R3_u, (TinyVector<1>{-1})), "error: incompatible operand types Vh(P0:R^3) and R^1");
        }

        SECTION("dot Rd*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<1>{3}), p_R1_u, dot);
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<2>{-6, 2}), p_R2_u, dot);
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<3>{-1, 5, 2}), p_R3_u, dot);
          REQUIRE_THROWS_WITH(dot((TinyVector<2>{-6, 2}), p_R1_u),
                              "error: incompatible operand types R^2 and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(dot((TinyVector<3>{-1, 5, 2}), p_R2_u),
                              "error: incompatible operand types R^3 and Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(dot((TinyVector<1>{-1}), p_R3_u), "error: incompatible operand types R^1 and Vh(P0:R^3)");
        }

        SECTION("sum_of_R* Vh -> R*")
        {
          REQUIRE(sum_of<double>(p_u) == sum(p_u->cellValues()));
          REQUIRE(sum_of<TinyVector<1>>(p_R1_u) == sum(p_R1_u->cellValues()));
          REQUIRE(sum_of<TinyVector<2>>(p_R2_u) == sum(p_R2_u->cellValues()));
          REQUIRE(sum_of<TinyVector<3>>(p_R3_u) == sum(p_R3_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<1>>(p_R1x1_u) == sum(p_R1x1_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<2>>(p_R2x2_u) == sum(p_R2x2_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<3>>(p_R3x3_u) == sum(p_R3x3_u->cellValues()));

          REQUIRE_THROWS_WITH(sum_of<TinyVector<1>>(p_u), "error: invalid operand type Vh(P0:R)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R2_u), "error: invalid operand type Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R3_u), "error: invalid operand type Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R1x1_u), "error: invalid operand type Vh(P0:R^1x1)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R2x2_u), "error: invalid operand type Vh(P0:R^2x2)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R3x3_u), "error: invalid operand type Vh(P0:R^3x3)");
        }

        SECTION("integral_of_R* Vh -> R*")
        {
          auto integrate_locally = [&](const auto& cell_values) {
            const auto& Vj = MeshDataManager::instance().getMeshData(*mesh).Vj();
            using DataType = decltype(double{} * cell_values[CellId{0}]);
            CellValue<DataType> local_integral{mesh->connectivity()};
            parallel_for(
              local_integral.numberOfItems(),
              PUGS_LAMBDA(const CellId cell_id) { local_integral[cell_id] = Vj[cell_id] * cell_values[cell_id]; });
            return local_integral;
          };

          REQUIRE(integral_of<double>(p_u) == sum(integrate_locally(p_u->cellValues())));
          REQUIRE(integral_of<TinyVector<1>>(p_R1_u) == sum(integrate_locally(p_R1_u->cellValues())));
          REQUIRE(integral_of<TinyVector<2>>(p_R2_u) == sum(integrate_locally(p_R2_u->cellValues())));
          REQUIRE(integral_of<TinyVector<3>>(p_R3_u) == sum(integrate_locally(p_R3_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<1>>(p_R1x1_u) == sum(integrate_locally(p_R1x1_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<2>>(p_R2x2_u) == sum(integrate_locally(p_R2x2_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<3>>(p_R3x3_u) == sum(integrate_locally(p_R3x3_u->cellValues())));

          REQUIRE_THROWS_WITH(integral_of<TinyVector<1>>(p_u), "error: invalid operand type Vh(P0:R)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R2_u), "error: invalid operand type Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R3_u), "error: invalid operand type Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R1x1_u), "error: invalid operand type Vh(P0:R^1x1)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R2x2_u), "error: invalid operand type Vh(P0:R^2x2)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R3x3_u), "error: invalid operand type Vh(P0:R^3x3)");
        }
      }
    }
  }

  SECTION("2D")
  {
    constexpr size_t Dimension = 2;

    using Rd = TinyVector<Dimension>;

    std::array mesh_list = MeshDataBaseForTests::get().all2DMeshes();

    for (auto named_mesh : mesh_list) {
      SECTION(named_mesh.name())
      {
        auto mesh = named_mesh.mesh();

        std::shared_ptr other_mesh =
          std::make_shared<Mesh<Connectivity<Dimension>>>(mesh->shared_connectivity(), mesh->xr());

        CellValue<const Rd> xj = MeshDataManager::instance().getMeshData(*mesh).xj();

        CellValue<double> values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 0.2 + std::cos(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        CellValue<double> positive_values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 2 + std::sin(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        CellValue<double> bounded_values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 0.9 * std::sin(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        std::shared_ptr p_u = std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, values);
        std::shared_ptr p_other_mesh_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(other_mesh, values);
        std::shared_ptr p_positive_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, positive_values);
        std::shared_ptr p_bounded_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, bounded_values);

        std::shared_ptr p_R1_u = [=] {
          CellValue<TinyVector<1>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) { uj[cell_id][0] = 2 * xj[cell_id][0] + 1; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(mesh, uj);
        }();

        std::shared_ptr p_R1_v = [=] {
          CellValue<TinyVector<1>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { vj[cell_id][0] = xj[cell_id][0] * xj[cell_id][0] + 1; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R1_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(other_mesh, p_R1_u->cellValues());

        constexpr auto to_2d = [&](const TinyVector<Dimension>& x) -> TinyVector<2> {
          if constexpr (Dimension == 1) {
            return TinyVector<2>{x[0], 1 + x[0] * x[0]};
          } else if constexpr (Dimension == 2) {
            return TinyVector<2>{x[0], x[1]};
          } else if constexpr (Dimension == 3) {
            return TinyVector<2>{x[0], x[1] + x[2]};
          }
        };

        std::shared_ptr p_R2_u = [=] {
          CellValue<TinyVector<2>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);
              uj[cell_id]           = TinyVector<2>{2 * x[0] + 1, 1 - x[1]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(mesh, uj);
        }();

        std::shared_ptr p_R2_v = [=] {
          CellValue<TinyVector<2>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);
              vj[cell_id]           = TinyVector<2>{x[0] * x[1] + 1, 2 * x[1]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R2_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(other_mesh, p_R2_u->cellValues());

        constexpr auto to_3d = [&](const TinyVector<Dimension>& x) -> TinyVector<3> {
          if constexpr (Dimension == 1) {
            return TinyVector<3>{x[0], 1 + x[0] * x[0], 2 - x[0]};
          } else if constexpr (Dimension == 2) {
            return TinyVector<3>{x[0], x[1], x[0] + x[1]};
          } else if constexpr (Dimension == 3) {
            return TinyVector<3>{x[0], x[1], x[2]};
          }
        };

        std::shared_ptr p_R3_u = [=] {
          CellValue<TinyVector<3>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              uj[cell_id]           = TinyVector<3>{2 * x[0] + 1, 1 - x[1] * x[2], x[0] + x[2]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(mesh, uj);
        }();

        std::shared_ptr p_R3_v = [=] {
          CellValue<TinyVector<3>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              vj[cell_id]           = TinyVector<3>{x[0] * x[1] + 1, 2 * x[1], x[2] * x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R3_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(other_mesh, p_R3_u->cellValues());

        std::shared_ptr p_R1x1_u = [=] {
          CellValue<TinyMatrix<1>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { uj[cell_id] = TinyMatrix<1>{2 * xj[cell_id][0] + 1}; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>>(mesh, uj);
        }();

        std::shared_ptr p_R2x2_u = [=] {
          CellValue<TinyMatrix<2>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);

              uj[cell_id] = TinyMatrix<2>{2 * x[0] + 1, 1 - x[1],   //
                                          2 * x[1], -x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>>(mesh, uj);
        }();

        std::shared_ptr p_R3x3_u = [=] {
          CellValue<TinyMatrix<3>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);

              uj[cell_id] = TinyMatrix<3>{2 * x[0] + 1,    1 - x[1],        3,             //
                                          2 * x[1],        -x[0],           x[0] - x[1],   //
                                          3 * x[2] - x[1], x[1] - 2 * x[2], x[2] - x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>>(mesh, uj);
        }();

        std::shared_ptr p_Vector3_u = [=] {
          CellArray<double> uj_vector{mesh->connectivity(), 3};
          parallel_for(
            uj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              uj_vector[cell_id][0] = 2 * x[0] + 1;
              uj_vector[cell_id][1] = 1 - x[1] * x[2];
              uj_vector[cell_id][2] = x[0] + x[2];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, uj_vector);
        }();

        std::shared_ptr p_Vector3_v = [=] {
          CellArray<double> vj_vector{mesh->connectivity(), 3};
          parallel_for(
            vj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              vj_vector[cell_id][0] = x[0] * x[1] + 1;
              vj_vector[cell_id][1] = 2 * x[1];
              vj_vector[cell_id][2] = x[2] * x[0];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, vj_vector);
        }();

        std::shared_ptr p_Vector2_w = [=] {
          CellArray<double> wj_vector{mesh->connectivity(), 2};
          parallel_for(
            wj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              wj_vector[cell_id][0] = x[0] + x[1] * 2;
              wj_vector[cell_id][1] = x[0] * x[1];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, wj_vector);
        }();

        SECTION("sqrt Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, sqrt);
          REQUIRE_THROWS_WITH(sqrt(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("abs Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, abs);
          REQUIRE_THROWS_WITH(abs(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("sin Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, sin);
          REQUIRE_THROWS_WITH(sin(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("cos Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, cos);
          REQUIRE_THROWS_WITH(cos(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("tan Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, tan);
          REQUIRE_THROWS_WITH(tan(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("asin Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, asin);
          REQUIRE_THROWS_WITH(asin(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("acos Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, acos);
          REQUIRE_THROWS_WITH(acos(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atan Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, atan);
          REQUIRE_THROWS_WITH(atan(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("sinh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, sinh);
          REQUIRE_THROWS_WITH(sinh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("cosh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, cosh);
          REQUIRE_THROWS_WITH(cosh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("tanh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, tanh);
          REQUIRE_THROWS_WITH(tanh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("asinh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, asinh);
          REQUIRE_THROWS_WITH(asinh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("acosh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, acosh);
          REQUIRE_THROWS_WITH(acosh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atanh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, atanh);
          REQUIRE_THROWS_WITH(atanh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("exp Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, exp);
          REQUIRE_THROWS_WITH(exp(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("log Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, log);
          REQUIRE_THROWS_WITH(log(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atan2 Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_positive_u, p_bounded_u, atan2);
          REQUIRE_THROWS_WITH(atan2(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(atan2(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
          REQUIRE_THROWS_WITH(atan2(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
        }

        SECTION("atan2 Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 3.6, atan2);
          REQUIRE_THROWS_WITH(atan2(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("atan2 R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(2.4, p_u, atan2);
          REQUIRE_THROWS_WITH(atan2(2.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("min Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_u, p_bounded_u, min);
          REQUIRE_THROWS_WITH(::min(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(::min(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(::min(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("min Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 1.2, min);
          REQUIRE_THROWS_WITH(min(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("min R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(0.4, p_u, min);
          REQUIRE_THROWS_WITH(min(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("min Vh -> R")
        {
          REQUIRE(min(std::shared_ptr<const IDiscreteFunction>{p_u}) == min(p_u->cellValues()));
          REQUIRE_THROWS_WITH(min(std::shared_ptr<const IDiscreteFunction>{p_R1_u}),
                              "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("max Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_u, p_bounded_u, max);
          REQUIRE_THROWS_WITH(::max(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(::max(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(::max(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("max Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 1.2, max);
          REQUIRE_THROWS_WITH(max(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("max Vh -> R")
        {
          REQUIRE(max(std::shared_ptr<const IDiscreteFunction>{p_u}) == max(p_u->cellValues()));
          REQUIRE_THROWS_WITH(max(std::shared_ptr<const IDiscreteFunction>{p_R1_u}),
                              "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("max R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(0.4, p_u, max);
          REQUIRE_THROWS_WITH(max(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("pow Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_positive_u, p_bounded_u, pow);
          REQUIRE_THROWS_WITH(pow(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(pow(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(pow(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("pow Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_positive_u, 3.3, pow);
          REQUIRE_THROWS_WITH(pow(p_R1_u, 3.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("pow R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(2.1, p_u, pow);
          REQUIRE_THROWS_WITH(pow(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("dot Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R1_u, p_R1_v, dot);
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R2_u, p_R2_v, dot);
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R3_u, p_R3_v, dot);

          {
            auto p_UV = dot(p_Vector3_u, p_Vector3_v);
            REQUIRE(p_UV.use_count() == 1);

            auto UV        = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*p_UV);
            auto direct_UV = dot(*p_Vector3_u, *p_Vector3_v);

            bool is_same = true;
            for (CellId cell_id = 0; cell_id < mesh->numberOfCells(); ++cell_id) {
              if (UV[cell_id] != direct_UV[cell_id]) {
                is_same = false;
                break;
              }
            }

            REQUIRE(is_same);
          }

          REQUIRE_THROWS_WITH(dot(p_R1_u, p_other_mesh_R1_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R2_u, p_other_mesh_R2_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R3_u, p_other_mesh_R3_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R1_u, p_R3_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(dot(p_Vector3_u, p_Vector2_w), "error: operands have different dimension");
        }

        SECTION("dot Vh*Rd -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R1_u, (TinyVector<1>{3}), dot);
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R2_u, (TinyVector<2>{-6, 2}), dot);
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R3_u, (TinyVector<3>{-1, 5, 2}), dot);
          REQUIRE_THROWS_WITH(dot(p_R1_u, (TinyVector<2>{-6, 2})),
                              "error: incompatible operand types Vh(P0:R^1) and R^2");
          REQUIRE_THROWS_WITH(dot(p_R2_u, (TinyVector<3>{-1, 5, 2})),
                              "error: incompatible operand types Vh(P0:R^2) and R^3");
          REQUIRE_THROWS_WITH(dot(p_R3_u, (TinyVector<1>{-1})), "error: incompatible operand types Vh(P0:R^3) and R^1");
        }

        SECTION("dot Rd*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<1>{3}), p_R1_u, dot);
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<2>{-6, 2}), p_R2_u, dot);
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<3>{-1, 5, 2}), p_R3_u, dot);
          REQUIRE_THROWS_WITH(dot((TinyVector<2>{-6, 2}), p_R1_u),
                              "error: incompatible operand types R^2 and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(dot((TinyVector<3>{-1, 5, 2}), p_R2_u),
                              "error: incompatible operand types R^3 and Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(dot((TinyVector<1>{-1}), p_R3_u), "error: incompatible operand types R^1 and Vh(P0:R^3)");
        }

        SECTION("sum_of_R* Vh -> R*")
        {
          REQUIRE(sum_of<double>(p_u) == sum(p_u->cellValues()));
          REQUIRE(sum_of<TinyVector<1>>(p_R1_u) == sum(p_R1_u->cellValues()));
          REQUIRE(sum_of<TinyVector<2>>(p_R2_u) == sum(p_R2_u->cellValues()));
          REQUIRE(sum_of<TinyVector<3>>(p_R3_u) == sum(p_R3_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<1>>(p_R1x1_u) == sum(p_R1x1_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<2>>(p_R2x2_u) == sum(p_R2x2_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<3>>(p_R3x3_u) == sum(p_R3x3_u->cellValues()));

          REQUIRE_THROWS_WITH(sum_of<TinyVector<1>>(p_u), "error: invalid operand type Vh(P0:R)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R2_u), "error: invalid operand type Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R3_u), "error: invalid operand type Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R1x1_u), "error: invalid operand type Vh(P0:R^1x1)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R2x2_u), "error: invalid operand type Vh(P0:R^2x2)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R3x3_u), "error: invalid operand type Vh(P0:R^3x3)");
        }

        SECTION("integral_of_R* Vh -> R*")
        {
          auto integrate_locally = [&](const auto& cell_values) {
            const auto& Vj = MeshDataManager::instance().getMeshData(*mesh).Vj();
            using DataType = decltype(double{} * cell_values[CellId{0}]);
            CellValue<DataType> local_integral{mesh->connectivity()};
            parallel_for(
              local_integral.numberOfItems(),
              PUGS_LAMBDA(const CellId cell_id) { local_integral[cell_id] = Vj[cell_id] * cell_values[cell_id]; });
            return local_integral;
          };

          REQUIRE(integral_of<double>(p_u) == sum(integrate_locally(p_u->cellValues())));
          REQUIRE(integral_of<TinyVector<1>>(p_R1_u) == sum(integrate_locally(p_R1_u->cellValues())));
          REQUIRE(integral_of<TinyVector<2>>(p_R2_u) == sum(integrate_locally(p_R2_u->cellValues())));
          REQUIRE(integral_of<TinyVector<3>>(p_R3_u) == sum(integrate_locally(p_R3_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<1>>(p_R1x1_u) == sum(integrate_locally(p_R1x1_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<2>>(p_R2x2_u) == sum(integrate_locally(p_R2x2_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<3>>(p_R3x3_u) == sum(integrate_locally(p_R3x3_u->cellValues())));

          REQUIRE_THROWS_WITH(integral_of<TinyVector<1>>(p_u), "error: invalid operand type Vh(P0:R)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R2_u), "error: invalid operand type Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R3_u), "error: invalid operand type Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R1x1_u), "error: invalid operand type Vh(P0:R^1x1)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R2x2_u), "error: invalid operand type Vh(P0:R^2x2)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R3x3_u), "error: invalid operand type Vh(P0:R^3x3)");
        }
      }
    }
  }

  SECTION("3D")
  {
    constexpr size_t Dimension = 3;

    using Rd = TinyVector<Dimension>;

    std::array mesh_list = MeshDataBaseForTests::get().all3DMeshes();

    for (auto named_mesh : mesh_list) {
      SECTION(named_mesh.name())
      {
        auto mesh = named_mesh.mesh();

        std::shared_ptr other_mesh =
          std::make_shared<Mesh<Connectivity<Dimension>>>(mesh->shared_connectivity(), mesh->xr());

        CellValue<const Rd> xj = MeshDataManager::instance().getMeshData(*mesh).xj();

        CellValue<double> values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 0.2 + std::cos(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        CellValue<double> positive_values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 2 + std::sin(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        CellValue<double> bounded_values = [=] {
          CellValue<double> build_values{mesh->connectivity()};
          parallel_for(
            build_values.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { build_values[cell_id] = 0.9 * std::sin(l2Norm(xj[cell_id])); });
          return build_values;
        }();

        std::shared_ptr p_u = std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, values);
        std::shared_ptr p_other_mesh_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(other_mesh, values);
        std::shared_ptr p_positive_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, positive_values);
        std::shared_ptr p_bounded_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, double>>(mesh, bounded_values);

        std::shared_ptr p_R1_u = [=] {
          CellValue<TinyVector<1>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) { uj[cell_id][0] = 2 * xj[cell_id][0] + 1; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(mesh, uj);
        }();

        std::shared_ptr p_R1_v = [=] {
          CellValue<TinyVector<1>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { vj[cell_id][0] = xj[cell_id][0] * xj[cell_id][0] + 1; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R1_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<1>>>(other_mesh, p_R1_u->cellValues());

        constexpr auto to_2d = [&](const TinyVector<Dimension>& x) -> TinyVector<2> {
          if constexpr (Dimension == 1) {
            return TinyVector<2>{x[0], 1 + x[0] * x[0]};
          } else if constexpr (Dimension == 2) {
            return TinyVector<2>{x[0], x[1]};
          } else if constexpr (Dimension == 3) {
            return TinyVector<2>{x[0], x[1] + x[2]};
          }
        };

        std::shared_ptr p_R2_u = [=] {
          CellValue<TinyVector<2>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);
              uj[cell_id]           = TinyVector<2>{2 * x[0] + 1, 1 - x[1]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(mesh, uj);
        }();

        std::shared_ptr p_R2_v = [=] {
          CellValue<TinyVector<2>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);
              vj[cell_id]           = TinyVector<2>{x[0] * x[1] + 1, 2 * x[1]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R2_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<2>>>(other_mesh, p_R2_u->cellValues());

        constexpr auto to_3d = [&](const TinyVector<Dimension>& x) -> TinyVector<3> {
          if constexpr (Dimension == 1) {
            return TinyVector<3>{x[0], 1 + x[0] * x[0], 2 - x[0]};
          } else if constexpr (Dimension == 2) {
            return TinyVector<3>{x[0], x[1], x[0] + x[1]};
          } else if constexpr (Dimension == 3) {
            return TinyVector<3>{x[0], x[1], x[2]};
          }
        };

        std::shared_ptr p_R3_u = [=] {
          CellValue<TinyVector<3>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              uj[cell_id]           = TinyVector<3>{2 * x[0] + 1, 1 - x[1] * x[2], x[0] + x[2]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(mesh, uj);
        }();

        std::shared_ptr p_R3_v = [=] {
          CellValue<TinyVector<3>> vj{mesh->connectivity()};
          parallel_for(
            vj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              vj[cell_id]           = TinyVector<3>{x[0] * x[1] + 1, 2 * x[1], x[2] * x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(mesh, vj);
        }();

        std::shared_ptr p_other_mesh_R3_u =
          std::make_shared<const DiscreteFunctionP0<Dimension, TinyVector<3>>>(other_mesh, p_R3_u->cellValues());

        std::shared_ptr p_R1x1_u = [=] {
          CellValue<TinyMatrix<1>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(),
            PUGS_LAMBDA(const CellId cell_id) { uj[cell_id] = TinyMatrix<1>{2 * xj[cell_id][0] + 1}; });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<1>>>(mesh, uj);
        }();

        std::shared_ptr p_R2x2_u = [=] {
          CellValue<TinyMatrix<2>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<2> x = to_2d(xj[cell_id]);

              uj[cell_id] = TinyMatrix<2>{2 * x[0] + 1, 1 - x[1],   //
                                          2 * x[1], -x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<2>>>(mesh, uj);
        }();

        std::shared_ptr p_R3x3_u = [=] {
          CellValue<TinyMatrix<3>> uj{mesh->connectivity()};
          parallel_for(
            uj.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);

              uj[cell_id] = TinyMatrix<3>{2 * x[0] + 1,    1 - x[1],        3,             //
                                          2 * x[1],        -x[0],           x[0] - x[1],   //
                                          3 * x[2] - x[1], x[1] - 2 * x[2], x[2] - x[0]};
            });

          return std::make_shared<const DiscreteFunctionP0<Dimension, TinyMatrix<3>>>(mesh, uj);
        }();

        std::shared_ptr p_Vector3_u = [=] {
          CellArray<double> uj_vector{mesh->connectivity(), 3};
          parallel_for(
            uj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              uj_vector[cell_id][0] = 2 * x[0] + 1;
              uj_vector[cell_id][1] = 1 - x[1] * x[2];
              uj_vector[cell_id][2] = x[0] + x[2];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, uj_vector);
        }();

        std::shared_ptr p_Vector3_v = [=] {
          CellArray<double> vj_vector{mesh->connectivity(), 3};
          parallel_for(
            vj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              vj_vector[cell_id][0] = x[0] * x[1] + 1;
              vj_vector[cell_id][1] = 2 * x[1];
              vj_vector[cell_id][2] = x[2] * x[0];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, vj_vector);
        }();

        std::shared_ptr p_Vector2_w = [=] {
          CellArray<double> wj_vector{mesh->connectivity(), 2};
          parallel_for(
            wj_vector.numberOfItems(), PUGS_LAMBDA(const CellId cell_id) {
              const TinyVector<3> x = to_3d(xj[cell_id]);
              wj_vector[cell_id][0] = x[0] + x[1] * 2;
              wj_vector[cell_id][1] = x[0] * x[1];
            });

          return std::make_shared<const DiscreteFunctionP0Vector<Dimension, double>>(mesh, wj_vector);
        }();

        SECTION("sqrt Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, sqrt);
          REQUIRE_THROWS_WITH(sqrt(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("abs Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, abs);
          REQUIRE_THROWS_WITH(abs(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("sin Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, sin);
          REQUIRE_THROWS_WITH(sin(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("cos Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, cos);
          REQUIRE_THROWS_WITH(cos(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("tan Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, tan);
          REQUIRE_THROWS_WITH(tan(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("asin Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, asin);
          REQUIRE_THROWS_WITH(asin(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("acos Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, acos);
          REQUIRE_THROWS_WITH(acos(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atan Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, atan);
          REQUIRE_THROWS_WITH(atan(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("sinh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, sinh);
          REQUIRE_THROWS_WITH(sinh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("cosh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, cosh);
          REQUIRE_THROWS_WITH(cosh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("tanh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, tanh);
          REQUIRE_THROWS_WITH(tanh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("asinh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, asinh);
          REQUIRE_THROWS_WITH(asinh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("acosh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, acosh);
          REQUIRE_THROWS_WITH(acosh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atanh Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_bounded_u, atanh);
          REQUIRE_THROWS_WITH(atanh(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("exp Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_u, exp);
          REQUIRE_THROWS_WITH(exp(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("log Vh -> Vh")
        {
          CHECK_EMBEDDED_VH_TO_VH_REAL_FUNCTION_EVALUATION(p_positive_u, log);
          REQUIRE_THROWS_WITH(log(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("atan2 Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_positive_u, p_bounded_u, atan2);
          REQUIRE_THROWS_WITH(atan2(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(atan2(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
          REQUIRE_THROWS_WITH(atan2(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
        }

        SECTION("atan2 Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 3.6, atan2);
          REQUIRE_THROWS_WITH(atan2(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("atan2 R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(2.4, p_u, atan2);
          REQUIRE_THROWS_WITH(atan2(2.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("min Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_u, p_bounded_u, min);
          REQUIRE_THROWS_WITH(::min(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(::min(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(::min(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("min Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 1.2, min);
          REQUIRE_THROWS_WITH(min(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("min R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(0.4, p_u, min);
          REQUIRE_THROWS_WITH(min(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("min Vh -> R")
        {
          REQUIRE(min(std::shared_ptr<const IDiscreteFunction>{p_u}) == min(p_u->cellValues()));
          REQUIRE_THROWS_WITH(min(std::shared_ptr<const IDiscreteFunction>{p_R1_u}),
                              "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("max Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_u, p_bounded_u, max);
          REQUIRE_THROWS_WITH(::max(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(::max(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(::max(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("max Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_u, 1.2, max);
          REQUIRE_THROWS_WITH(max(p_R1_u, 2.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("max Vh -> R")
        {
          REQUIRE(max(std::shared_ptr<const IDiscreteFunction>{p_u}) == max(p_u->cellValues()));
          REQUIRE_THROWS_WITH(max(std::shared_ptr<const IDiscreteFunction>{p_R1_u}),
                              "error: invalid operand type Vh(P0:R^1)");
        }

        SECTION("max R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(0.4, p_u, max);
          REQUIRE_THROWS_WITH(max(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("pow Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_positive_u, p_bounded_u, pow);
          REQUIRE_THROWS_WITH(pow(p_u, p_other_mesh_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(pow(p_u, p_R1_u), "error: incompatible operand types Vh(P0:R) and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(pow(p_R1_u, p_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R)");
        }

        SECTION("pow Vh*R -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_positive_u, 3.3, pow);
          REQUIRE_THROWS_WITH(pow(p_R1_u, 3.1), "error: incompatible operand types Vh(P0:R^1) and R");
        }

        SECTION("pow R*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION(2.1, p_u, pow);
          REQUIRE_THROWS_WITH(pow(3.1, p_R1_u), "error: incompatible operand types R and Vh(P0:R^1)");
        }

        SECTION("dot Vh*Vh -> Vh")
        {
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R1_u, p_R1_v, dot);
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R2_u, p_R2_v, dot);
          CHECK_EMBEDDED_VH2_TO_VH_FUNCTION_EVALUATION(p_R3_u, p_R3_v, dot);

          {
            auto p_UV = dot(p_Vector3_u, p_Vector3_v);
            REQUIRE(p_UV.use_count() == 1);

            auto UV        = dynamic_cast<const DiscreteFunctionP0<Dimension, double>&>(*p_UV);
            auto direct_UV = dot(*p_Vector3_u, *p_Vector3_v);

            bool is_same = true;
            for (CellId cell_id = 0; cell_id < mesh->numberOfCells(); ++cell_id) {
              if (UV[cell_id] != direct_UV[cell_id]) {
                is_same = false;
                break;
              }
            }

            REQUIRE(is_same);
          }

          REQUIRE_THROWS_WITH(dot(p_R1_u, p_other_mesh_R1_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R2_u, p_other_mesh_R2_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R3_u, p_other_mesh_R3_u), "error: operands are defined on different meshes");
          REQUIRE_THROWS_WITH(dot(p_R1_u, p_R3_u), "error: incompatible operand types Vh(P0:R^1) and Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(dot(p_Vector3_u, p_Vector2_w), "error: operands have different dimension");
        }

        SECTION("dot Vh*Rd -> Vh")
        {
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R1_u, (TinyVector<1>{3}), dot);
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R2_u, (TinyVector<2>{-6, 2}), dot);
          CHECK_EMBEDDED_VHxW_TO_VH_FUNCTION_EVALUATION(p_R3_u, (TinyVector<3>{-1, 5, 2}), dot);
          REQUIRE_THROWS_WITH(dot(p_R1_u, (TinyVector<2>{-6, 2})),
                              "error: incompatible operand types Vh(P0:R^1) and R^2");
          REQUIRE_THROWS_WITH(dot(p_R2_u, (TinyVector<3>{-1, 5, 2})),
                              "error: incompatible operand types Vh(P0:R^2) and R^3");
          REQUIRE_THROWS_WITH(dot(p_R3_u, (TinyVector<1>{-1})), "error: incompatible operand types Vh(P0:R^3) and R^1");
        }

        SECTION("dot Rd*Vh -> Vh")
        {
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<1>{3}), p_R1_u, dot);
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<2>{-6, 2}), p_R2_u, dot);
          CHECK_EMBEDDED_WxVH_TO_VH_FUNCTION_EVALUATION((TinyVector<3>{-1, 5, 2}), p_R3_u, dot);
          REQUIRE_THROWS_WITH(dot((TinyVector<2>{-6, 2}), p_R1_u),
                              "error: incompatible operand types R^2 and Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(dot((TinyVector<3>{-1, 5, 2}), p_R2_u),
                              "error: incompatible operand types R^3 and Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(dot((TinyVector<1>{-1}), p_R3_u), "error: incompatible operand types R^1 and Vh(P0:R^3)");
        }

        SECTION("sum_of_R* Vh -> R*")
        {
          REQUIRE(sum_of<double>(p_u) == sum(p_u->cellValues()));
          REQUIRE(sum_of<TinyVector<1>>(p_R1_u) == sum(p_R1_u->cellValues()));
          REQUIRE(sum_of<TinyVector<2>>(p_R2_u) == sum(p_R2_u->cellValues()));
          REQUIRE(sum_of<TinyVector<3>>(p_R3_u) == sum(p_R3_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<1>>(p_R1x1_u) == sum(p_R1x1_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<2>>(p_R2x2_u) == sum(p_R2x2_u->cellValues()));
          REQUIRE(sum_of<TinyMatrix<3>>(p_R3x3_u) == sum(p_R3x3_u->cellValues()));

          REQUIRE_THROWS_WITH(sum_of<TinyVector<1>>(p_u), "error: invalid operand type Vh(P0:R)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R2_u), "error: invalid operand type Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R3_u), "error: invalid operand type Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R1x1_u), "error: invalid operand type Vh(P0:R^1x1)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R2x2_u), "error: invalid operand type Vh(P0:R^2x2)");
          REQUIRE_THROWS_WITH(sum_of<double>(p_R3x3_u), "error: invalid operand type Vh(P0:R^3x3)");
        }

        SECTION("integral_of_R* Vh -> R*")
        {
          auto integrate_locally = [&](const auto& cell_values) {
            const auto& Vj = MeshDataManager::instance().getMeshData(*mesh).Vj();
            using DataType = decltype(double{} * cell_values[CellId{0}]);
            CellValue<DataType> local_integral{mesh->connectivity()};
            parallel_for(
              local_integral.numberOfItems(),
              PUGS_LAMBDA(const CellId cell_id) { local_integral[cell_id] = Vj[cell_id] * cell_values[cell_id]; });
            return local_integral;
          };

          REQUIRE(integral_of<double>(p_u) == sum(integrate_locally(p_u->cellValues())));
          REQUIRE(integral_of<TinyVector<1>>(p_R1_u) == sum(integrate_locally(p_R1_u->cellValues())));
          REQUIRE(integral_of<TinyVector<2>>(p_R2_u) == sum(integrate_locally(p_R2_u->cellValues())));
          REQUIRE(integral_of<TinyVector<3>>(p_R3_u) == sum(integrate_locally(p_R3_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<1>>(p_R1x1_u) == sum(integrate_locally(p_R1x1_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<2>>(p_R2x2_u) == sum(integrate_locally(p_R2x2_u->cellValues())));
          REQUIRE(integral_of<TinyMatrix<3>>(p_R3x3_u) == sum(integrate_locally(p_R3x3_u->cellValues())));

          REQUIRE_THROWS_WITH(integral_of<TinyVector<1>>(p_u), "error: invalid operand type Vh(P0:R)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R1_u), "error: invalid operand type Vh(P0:R^1)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R2_u), "error: invalid operand type Vh(P0:R^2)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R3_u), "error: invalid operand type Vh(P0:R^3)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R1x1_u), "error: invalid operand type Vh(P0:R^1x1)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R2x2_u), "error: invalid operand type Vh(P0:R^2x2)");
          REQUIRE_THROWS_WITH(integral_of<double>(p_R3x3_u), "error: invalid operand type Vh(P0:R^3x3)");
        }
      }
    }
  }
}
